<?php

namespace App\Http\Controllers;

use App\DataTables\UsersDataTable;
use App\Helpers\Helper;
use App\Imports\UsersImport;
use App\Jobs\KirimNotifWa;
use App\Models\Dpt;
use App\Models\KabupatenKota;
use App\Models\Kecamatan;
use App\Models\KelurahanDesa;
use App\Models\Provinsi;
use App\Models\Tps;
use App\Models\User;
use App\Models\Partai;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    // public function index()
    // {   
    //     $q = request()->query('q');
    //     $datas = User::with('partai','tps.kelurahan_desa','kelurahan_desa')->where(function($w)use($q){
    //         $w->where('nama','like','%'.$q.'%')->orWhere('no_telp','like','%'.$q.'%');
    //     })->where("role","!=","saksi");
    //     if(auth()->user()->id != 1){
    //         $datas = $datas->where('id','!=',1);
    //     }
    //     $datas = $datas->orderBy('created_at','DESC')->paginate(10)->appends(['q'=>$q]);
    //     return view('user.index',compact('datas','q'));
    // }

    public function index(UsersDataTable $dataTable)
    {   
        if(auth()->user()->role != "admin"){
            return abort(404);
        }
        $provinsis = Provinsi::all();
        return $dataTable->render('user.index',compact('provinsis'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {   
        $provinsis = Provinsi::all();
        $partais = Partai::all();
        $password = substr(str_shuffle('0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ'), 1, 8);
        return view('user.create',compact('provinsis','partais','password'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'nama' => 'required',
            // 'no_telp' => 'required|unique:users,no_telp,NULL,NULL,deleted_at,NULL',
            'no_wa' => 'required|unique:users,no_telp,NULL,NULL,deleted_at,NULL',
            'password' => 'required|min:5',
            'role' => 'required',
        ]);

        DB::beginTransaction();
        try {
            $status = "sesuai";
            if($request->dpt_id){
                $cekUser = User::where('dpt_id', $request->dpt_id)->where('tps_id', $request->tps_id)->first();
                if ($cekUser && $request->role != "admin") {
                    $status = "sudah ada";
                }
            } else {
                $status = "tidak sesuai";
            }
            $no_wa = str_replace("'","",$request->no_wa);
            //remove all symbol and alphabet from $no_wa
            $no_wa = preg_replace('/[^\p{L}\p{N}\s]/u', '', $no_wa);

            //check if first character not 0
            if(mb_substr($no_wa, 0, 1) != '0'){
                $no_wa = '0'.$no_wa;
            }
            $user = User::create([
                'nama'=>$request->nama,
                'email'=>$request->email,
                'password'=>bcrypt($request->password),
                'role'=>$request->role,
                'no_telp'=>$no_wa,
                'no_wa'=>$no_wa,
                'provinsi_id' => $request->provinsi_id,
                'kabupaten_kota_id' => $request->kabupaten_kota_id,
                'kecamatan_id' => $request->kecamatan_id,
                'kelurahan_desa_id' => $request->kelurahan_desa_id,
                'tps_id' => $request->tps_id,
                'dpt_id' => $request->dpt_id,
                'status' => $status,
                'partai_id'=>$request->partai_id,
            ]);
            DB::commit();
            if ($request->has('no_wa') && env("APP_ENV") != "local") {
                $msg = 'Anda telah terdaftar sebagai '.$user->role.' untuk aplikasi *Pilkada ' . env("APP_NAME") . '* : 
Silahkan login di ' . route('login');
                Helper::sendWa($request->no_wa, $msg);
            }
            Helper::addUserLog("Menambah user ".$user->nama);
            return redirect()->route('user.index')->with('success','Data berhasil disimpan : '.$user->nama);
        } catch (\Throwable $th) {
            Log::error($th->getMessage());
            DB::rollback();
            return redirect()->back()->with('error','Data gagal disimpan')->withInput($request->all());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        return view('user.show',compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {   
        if(auth()->user()->id != 1 && $user->id == 1){
            return abort(404);
        }
        $provinsis = Provinsi::all();
        $kabupatenKotas = KabupatenKota::all();
        $kecamatans = Kecamatan::all();
        $kelurahanDesas = KelurahanDesa::all();
        $tps = Tps::with('kelurahan_desa')->get();
        $partais = Partai::all();
        return view('user.edit',['data'=>$user ,'provinsis'=>$provinsis,'kabupatenKotas'=>$kabupatenKotas,'kecamatans'=>$kecamatans,'kelurahanDesas'=>$kelurahanDesas,'tps'=>$tps,'partais'=>$partais]);   
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {   
        if(auth()->user()->id != 1 && $user->id == 1){
            return abort(404);
        }
        $request->validate([
            'nama' => 'required',
            // 'no_telp' => 'required|unique:users,no_telp,'.$user->id.',id,deleted_at,NULL',
            'no_wa' => 'required|unique:users,no_wa,'.$user->id.',id,deleted_at,NULL',
            'role' => 'required',
            
        ]);

        DB::beginTransaction();
        try {
            $status = $user->status;
            if($request->dpt_id){
                if($status == "sudah ada"){
                    $cekUser = User::where('dpt_id', $request->dpt_id)
                    ->where('id','!=',$user->id)
                    ->where('tps_id', $request->tps_id)->first();
                    if (!$cekUser) {
                        $status = "sesuai";
                    }
                }
                if($status == "tidak sesuai" && $request->dpt_id){
                    $cekDpt = Dpt::where('id',$request->dpt_id)->where('tps_id',$request->tps_id)->first();
                    if($cekDpt){
                        $status = "sesuai";
                    }
                }
            } else {
                $status = "tidak sesuai";
            }
            $no_wa = str_replace("'","",$request->no_wa);
            //remove all symbol and alphabet from $no_wa
            $no_wa = preg_replace('/[^\p{L}\p{N}\s]/u', '', $no_wa);

            //check if first character not 0
            if(mb_substr($no_wa, 0, 1) != '0'){
                $no_wa = '0'.$no_wa;
            }
            $user->update([
                'nama'=>$request->nama,
                'email'=>$request->email,
                'role'=>$request->role,
                'no_telp'=>$no_wa,
                'no_wa'=>$no_wa,
                'provinsi_id'=>$request->provinsi_id,
                'kabupaten_kota_id'=>$request->kabupaten_kota_id,
                'kecamatan_id'=>$request->kecamatan_id,
                'kelurahan_desa_id'=>$request->kelurahan_desa_id,
                'tps_id'=>$request->tps_id,
                'partai_id'=>$request->partai_id,
                'status'=>$status
            ]);

            if($request->password_baru){
                $request->validate([
                    'password_baru' => 'required|min:5',
                ]);
                $user->update([
                    'password'=>bcrypt($request->password_baru)
                ]);
            }
            DB::commit();
            Helper::addUserLog("Mengubah user ".$user->nama);
            return redirect()->route('user.index')->with('success','Data berhasil diubah : '.$user->nama);
        } catch (\Throwable $th) {
            Log::error($th->getMessage());
            DB::rollback();
            return redirect()->back()->with('error','Data gagal diubah')->withInput($request->all());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {   
        if(auth()->user()->id != 1 && $user->id == 1){
            return abort(404);
        }
        $suara = $user->suara;
        if($suara){
            return redirect()->back()->with('warning','Data tidak bisa dihapus, data ini masih dipakai di data lain'); 
        }
        DB::beginTransaction();
        try {
            $user->delete();
        
            $cekUser = User::where('dpt_id', $user->dpt_id)->where('status','sudah ada')->where('id','!=',$user->id)->where('tps_id', $user->tps_id)->count();
            if($cekUser == 1){
                User::where('dpt_id', $user->dpt_id)->where('status','sudah ada')->where('id','!=',$user->id)->where('tps_id', $user->tps_id)->update([
                    'status'=>'sesuai'
                ]);
            }
            DB::commit();
            Helper::addUserLog("Menghapus user ".$user->nama);
            return redirect()->route('user.index')->with('success','Data berhasil dihapus : '.$user->nama);
        } catch (\Throwable $th) {
            Log::error($th->getMessage());
            DB::rollback();
            return redirect()->route('user.index')->with('error','Data gagal dihapus');
        }
    }

    public function importView()
    {
        return view('user.import');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls'
        ]);

        $file = $request->file('file');
        DB::beginTransaction();
        try {
            $model = new UsersImport;
            $import = Excel::import($model, $file);
            DB::commit();
            $users = $model->users;
            
            foreach ($users as $user) {
                if($user['no_wa'] == null || $user['no_wa'] == "" || env("APP_ENV") == "local") continue;
                $msg = 'Anda telah terdaftar sebagai admin untuk aplikasi *Pilkada ' . env("APP_NAME") . '* : 
Silahkan login di ' . route('login');
                dispatch(new KirimNotifWa($user['no_wa'],$msg));
            }
            Helper::addUserLog("Mengimport data ".count($users)." user");
            return redirect()->route('user.index')->with('success', 'berhasil mengimport '.count($users).' user');
        } catch (\Throwable $th) {
            DB::rollBack();
            Log::error($th);
            return redirect()->back()->with('error', 'Data gagal diimport : ' . $th->getMessage());
        }
    }
}
