<?php

namespace App\Http\Controllers;

use App\DataTables\LaporanPelatihansDataTable;
use App\DataTables\PelatihsDataTable;
use App\Exports\FormatImportPelatihsExport;
use App\Helpers\Helper;
use App\Imports\PelatihsImport;
use App\Models\Pelatih;
use App\Models\Provinsi;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

class PelatihController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(PelatihsDataTable $pelatihsDataTable,LaporanPelatihansDataTable $laporanPelatihansDataTable)
    {
        $provinsis = Provinsi::all();
        // return $dataTable->render('pelatih.index', compact('provinsis'));
        return view('pelatih.index', [
            'provinsis'=>$provinsis,
            'pelatihsDataTable' => $pelatihsDataTable->html(),
            'laporanPelatihansDataTable' => $laporanPelatihansDataTable->html()
        ]);
    }

    //Gets Users JSON

    public function getPelatih(PelatihsDataTable $dataTable)
    {
        return $dataTable->render('pelatih.index');
    }

    //Gets Products JSON

    public function getLaporanPelatihan(LaporanPelatihansDataTable $dataTable)
    {
        return $dataTable->render('pelatih.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $provinsis = Provinsi::all();
        return view('pelatih.create', compact('provinsis'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'nama' => 'required',
            'provinsi_id' => 'required',
            'kabupaten_kota_id' => 'required'
        ]);

        DB::beginTransaction();
        try {

            $no_wa = str_replace("'", "", $request->no_wa);
            //remove all symbol and alphabet from $no_wa
            $no_wa = preg_replace('/[^\p{L}\p{N}\s]/u', '', $no_wa);

            //check if first character not 0
            if (mb_substr($no_wa, 0, 1) != '0') {
                $no_wa = '0' . $no_wa;
            }
            $pelatih = Pelatih::create([
                'nama' => $request->nama,
                'email' => $request->email,
                'no_wa' => $no_wa,
                'provinsi_id' => $request->provinsi_id,
                'kabupaten_kota_id' => $request->kabupaten_kota_id,
                'kecamatan_id' => $request->kecamatan_id,
                'kelurahan_desa_id' => $request->kelurahan_desa_id,
            ]);
            DB::commit();

            Helper::addUserLog("Menambah pelatih " . $pelatih->nama);
            return redirect()->route('pelatih.index')->with('success', 'Data berhasil disimpan : ' . $pelatih->nama);
        } catch (\Throwable $th) {
            Log::error($th->getMessage());
            DB::rollback();
            return redirect()->back()->with('error', 'Data gagal disimpan')->withInput($request->all());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Pelatih $pelatih)
    {

    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Pelatih $pelatih)
    {
        $provinsis = Provinsi::all();
        return view('pelatih.edit', ['data' => $pelatih, 'provinsis' => $provinsis]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Pelatih $pelatih)
    {
        $request->validate([
            'nama' => 'required',
            'provinsi_id' => 'required',
            'kabupaten_kota_id' => 'required'
        ]);

        DB::beginTransaction();
        try {

            $no_wa = str_replace("'", "", $request->no_wa);
            //remove all symbol and alphabet from $no_wa
            $no_wa = preg_replace('/[^\p{L}\p{N}\s]/u', '', $no_wa);

            //check if first character not 0
            if (mb_substr($no_wa, 0, 1) != '0') {
                $no_wa = '0' . $no_wa;
            }
            $pelatih->update([
                'nama' => $request->nama,
                'email' => $request->email,
                'no_wa' => $no_wa,
                'provinsi_id' => $request->provinsi_id,
                'kabupaten_kota_id' => $request->kabupaten_kota_id,
                'kecamatan_id' => $request->kecamatan_id,
                'kelurahan_desa_id' => $request->kelurahan_desa_id,
            ]);
            DB::commit();

            Helper::addUserLog("Mengubah pelatih " . $pelatih->nama);
            return redirect()->route('pelatih.index')->with('success', 'Data berhasil disimpan : ' . $pelatih->nama);
        } catch (\Throwable $th) {
            Log::error($th->getMessage());
            DB::rollback();
            return redirect()->back()->with('error', 'Data gagal disimpan')->withInput($request->all());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Pelatih $pelatih)
    {
        DB::beginTransaction();
        try {

            $pelatih->delete();
            DB::commit();

            Helper::addUserLog("Menghapus pelatih " . $pelatih->nama);
            return redirect()->route('pelatih.index')->with('success', 'Data berhasil dihapus : ' . $pelatih->nama);
        } catch (\Throwable $th) {
            Log::error($th->getMessage());
            DB::rollback();
            return redirect()->back()->with('error', 'Data gagal dihapus');
        }
    }

    public function importView()
    {
        if (request()->export) {
            $exportClass = new FormatImportPelatihsExport(auth()->user());
            return Excel::download($exportClass, 'import_pelatih_' . date("YmdHis") . '.xlsx');
        }
        return view('pelatih.import');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls'
        ]);

        $file = $request->file('file');
        DB::beginTransaction();
        try {
            $model = new PelatihsImport(Provinsi::with('kabupaten_kotas.kecamatans.kelurahan_desas')->get());
            $import = Excel::import($model, $file);
            DB::commit();
            Helper::addUserLog("Mengimport data pelatih");
            return redirect()->route('pelatih.index')->with('success', 'berhasil mengimport pelatih');
        } catch (\Throwable $th) {
            DB::rollBack();
            Log::error($th);
            return redirect()->back()->with('error', 'Data gagal diimport : ' . $th->getMessage());
        }

    }
}
