<?php

namespace App\Http\Controllers;

use App\Helpers\Helper;
use App\Models\LaporanPelatihan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class LaporanPelatihanController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('laporan-pelatihan.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'tanggal_kegiatan' => 'required|date',
            'uraian_kegiatan' => 'required',
            'lokasi' => 'required'
        ]);

        DB::beginTransaction();
        try {
            
            $laporan = LaporanPelatihan::create([
                'user_id'=>auth()->user()->id,
                'tanggal_kegiatan'=>$request->tanggal_kegiatan,
                'uraian_kegiatan'=>$request->uraian_kegiatan,
                'lokasi'=>$request->lokasi
            ]);

            $file_absensis = [];
            if($request->hasFile('file_absensi')){
                foreach ($request->file_absensi as $file_absensi) {
                    $file_absensis[] = $file_absensi->store('file-absensi/'.$laporan->id);
                }
            }

            $file_dokumentasis = [];
            if($request->hasFile('file_dokumentasi')){
                foreach ($request->file_dokumentasi as $file_dokumentasi) {
                    $file_dokumentasis[] = $file_dokumentasi->store('file-dokumentasi/'.$laporan->id);
                }
            }

            $file_anggarans = [];
            if($request->hasFile('file_anggaran')){
                foreach ($request->file_anggaran as $file_anggaran) {
                    $file_anggarans[] = $file_anggaran->store('file-anggaran/'.$laporan->id);
                }
            }

            $laporan->update([
                'file_absensi'=>implode(';',$file_absensis),
                'file_dokumentasi'=>implode(';',$file_dokumentasis),
                'file_anggaran'=>implode(';',$file_anggarans),
            ]);
            DB::commit();
            Helper::addUserLog("Menambah laporan pelatihan ".$laporan->tanggal_kegiatan);
            return redirect()->route('pelatih.index')->with('success','Data laporan berhasil disimpan');
        } catch (\Throwable $th) {
            DB::rollback();
            Log::error($th->getMessage());
            return redirect()->back()->withInput($request->all())->with('error','Data laporan gagal disimpan');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(LaporanPelatihan $laporanPelatihan)
    {
        return view('laporan-pelatihan.show',['laporan'=>$laporanPelatihan]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(LaporanPelatihan $laporanPelatihan)
    {
        return view('laporan-pelatihan.edit',['data'=>$laporanPelatihan]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, LaporanPelatihan $laporanPelatihan)
    {
        $request->validate([
            'tanggal_kegiatan' => 'required|date',
            'uraian_kegiatan' => 'required',
            'lokasi' => 'required'
        ]);

        DB::beginTransaction();
        try {
            
            $laporanPelatihan->update([
                'tanggal_kegiatan'=>$request->tanggal_kegiatan,
                'uraian_kegiatan'=>$request->uraian_kegiatan,
                'lokasi'=>$request->lokasi
            ]);

            $file_absensis = $request->file_absensi_old ?? [];
            if($request->hasFile('file_absensi')){
                foreach ($request->file_absensi as $file_absensi) {
                    $file_absensis[] = $file_absensi->store('file-absensi/'.$laporanPelatihan->id);
                }
            }

            $file_dokumentasis = $request->file_dokumentasi_old ?? [];
            if($request->hasFile('file_dokumentasi')){
                foreach ($request->file_dokumentasi as $file_dokumentasi) {
                    $file_dokumentasis[] = $file_dokumentasi->store('file-dokumentasi/'.$laporanPelatihan->id);
                }
            }

            $file_anggarans = $request->file_anggaran_old ?? [];
            if($request->hasFile('file_anggaran')){
                foreach ($request->file_anggaran as $file_anggaran) {
                    $file_anggarans[] = $file_anggaran->store('file-anggaran/'.$laporanPelatihan->id);
                }
            }

            $laporanPelatihan->update([
                'file_absensi'=>implode(';',$file_absensis),
                'file_dokumentasi'=>implode(';',$file_dokumentasis),
                'file_anggaran'=>implode(';',$file_anggarans),
            ]);
            DB::commit();
            Helper::addUserLog("Mengubah laporan pelatihan ".$laporanPelatihan->tanggal_kegiatan);
            return redirect()->route('pelatih.index')->with('success','Data laporan berhasil disimpan');
        } catch (\Throwable $th) {
            DB::rollback();
            Log::error($th->getMessage());
            return redirect()->back()->withInput($request->all())->with('error','Data laporan gagal disimpan');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(LaporanPelatihan $laporanPelatihan)
    {
        DB::beginTransaction();
        try {
            
            $laporanPelatihan->delete();
            DB::commit();
            Helper::addUserLog("Menghapus laporan pelatihan ".$laporanPelatihan->tanggal_kegiatan);
            return redirect()->route('pelatih.index')->with('success','Data laporan berhasil dihapus');
        } catch (\Throwable $th) {
            DB::rollback();
            Log::error($th->getMessage());
            return redirect()->back()->with('error','Data laporan gagal dihapus');
        }
    }

    public function getFileAbsen($idLaporan,$index){
        $laporanPelatihan = LaporanPelatihan::findOrFail($idLaporan);

     
        $path = $laporanPelatihan->list_file_absensi[$index];

        if (Storage::exists($path)) {
            //get mimes type
            $mimeType = Storage::mimeType($path);
            //get file content
            $file = Storage::get($path);
            //return file
            return response($file, 200)->header('Content-Type', $mimeType);
        } else {
            return abort(404);
        }
    }

    public function getFileDokumentasi($idLaporan,$index){
        $laporanPelatihan = LaporanPelatihan::findOrFail($idLaporan);
        
     
        $path = $laporanPelatihan->list_file_dokumentasi[$index];

        if (Storage::exists($path)) {
            //get mimes type
            $mimeType = Storage::mimeType($path);
            //get file content
            $file = Storage::get($path);
            //return file
            return response($file, 200)->header('Content-Type', $mimeType);
        } else {
            return abort(404);
        }
    }

    public function getFileAnggaran($idLaporan,$index){
        $laporanPelatihan = LaporanPelatihan::findOrFail($idLaporan);
        
     
        $path = $laporanPelatihan->list_file_anggaran[$index];

        if (Storage::exists($path)) {
            //get mimes type
            $mimeType = Storage::mimeType($path);
            //get file content
            $file = Storage::get($path);
            //return file
            return response($file, 200)->header('Content-Type', $mimeType);
        } else {
            return abort(404);
        }
    }
}
