<?php

namespace App\DataTables;

use App\Models\User;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class UsersDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addIndexColumn()
            ->addColumn('action', function ($data) {
                return view('user.action', compact('data'));
            })
            ->editColumn('status', function ($data) {
                return $data->status == "sudah ada"? "Sudah Ada" : "Sesuai";
            })
            ->setRowClass(function ($data) {
                return $data->status_color != 'green' ? 'text-'.$data->status_color.'-500' : '';
            })
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(User $model): QueryBuilder
    {
        $datas = $model->select('users.*')->with('partai', 'tps','tps.kelurahan_desa', 'kelurahan_desa')->where("role", "!=", "saksi");
        if (auth()->user()->id != 1) {
            $datas = $datas->where('id', '!=', 1);
        }

        return $datas;
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()->scrollX(true)
            ->setTableId('users-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            //->dom('Bfrtip')
            ->orderBy(1)
            ->selectStyleSingle()
            ->buttons([
                Button::make('excel'),
                Button::make('csv'),
                Button::make('pdf'),
                Button::make('print'),
                Button::make('reset'),
                Button::make('reload')
            ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('DT_RowIndex')
                ->title('#')
                ->searchable(false)
                ->orderable(false)
                ->width(10)
                ->addClass('text-center'),
            Column::make('nama'),
            Column::make('role'),
            Column::make('no_wa'),
            // Column::make('tps.nomor')->title("TPS"),
            Column::make('tps.kelurahan_desa.nama')->title('Kelurahan/Desa'),
            // Column::make('partai.nama')->title('Partai'),
            Column::make('status'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Users_' . date('YmdHis');
    }
}
