<?php

namespace App\DataTables;

use App\Models\User;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class SaksisDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
        ->addIndexColumn()
        ->addColumn('action', function ($data) {
            return view('saksi.action', compact('data'));
        })
        ->editColumn('status', function ($data) {
            return $data->status == "sudah ada"? "Sudah Ada" : "Sesuai";
        })
        ->filterColumn('suaras_count', function($query, $keyword) {
            if(strtolower($keyword) == "belum"){
                $query->having("suaras_count",'=',0);
            }
            if(strtolower($keyword) == "sudah"){
                $query->having("suaras_count",'>',0);
            }
        })
        ->editColumn('suaras_count', function ($data) {
            return $data->suaras_count == 0? "Belum" : "Sudah";
        })
        
        ->setRowClass(function ($data) {
            return $data->status_color != 'green' ? 'text-'.$data->status_color.'-500' : '';
        })
        ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(User $model): QueryBuilder
    {
        $datas = $model->select('users.*')->with('tps','tps.kelurahan_desa', 'kelurahan_desa', 'partai', 'dpt')
            ->withCount('suaras')
            ->where('role', 'saksi');
        
        if (request()->koordinator) {
            $datas = $datas->where('role', 'koordinator-' . request()->koordinator);
        }

        if (auth()->user()->role == "koordinator-kelurahan-desa") {
            $datas = $datas->whereHas('tps', function ($w) {
                $w->where('tps.kelurahan_desa_id', auth()->user()->kelurahan_desa_id);
            });
        }

        if (auth()->user()->role == "koordinator-kecamatan") {
            $datas = $datas->whereHas('tps.kelurahan_desa', function ($w) {
                $w->where('kelurahan_desas.kecamatan_id', auth()->user()->kecamatan_id);
            });
        }

        if (auth()->user()->role == "koordinator-kabupaten-kota") {
            $datas = $datas->whereHas('tps.kelurahan_desa.kecamatan', function ($w) {
                $w->where('kecamatans.kabupaten_kota_id', auth()->user()->kabupaten_kota_id);
            });
        }

        if (request()->kelurahan_desa_id) {
            $datas = $datas->where('users.kelurahan_desa_id', request()->kelurahan_desa_id);
        } else if (request()->kecamatan_id) {
            $datas = $datas->whereHas('kelurahan_desa', function ($w) {
                $w->where('kelurahan_desas.kecamatan_id', request()->kecamatan_id);
            });
        } else if (request()->kabupaten_kota_id) {
            $datas = $datas->whereHas('kelurahan_desa', function ($w) {
                $w->whereHas('kecamatan', function ($w) {
                    $w->where('kecamatans.kabupaten_kota_id', request()->kabupaten_kota_id);
                });
            });
        } else if (request()->provinsi_id) {
            $datas = $datas->whereHas('kelurahan_desa', function ($w) {
                $w->whereHas('kecamatan', function ($w) {
                    $w->whereHas('kabupaten_kota', function ($w) {
                        $w->where('kabupaten_kotas.provinsi_id', request()->provinsi_id);
                    });
                });
            });
        }

        if (request()->partai_id) {
            $datas = $datas->where('partai_id', request()->partai_id);
        }


        if (request()->input_suara) {
            $keyword = request()->input_suara;
            if(strtolower($keyword) == "belum"){
                $datas = $datas->having("suaras_count",'=',0);
            }
            if(strtolower($keyword) == "sudah"){
                $datas = $datas->having("suaras_count",'>',0);
            }
        }
        
        
        return $datas;
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()->scrollX(true)
                    ->setTableId('saksis-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                    ->orderBy(1)
                    ->selectStyleSingle()
                    ->buttons([
                        Button::make('excel'),
                        Button::make('csv'),
                        Button::make('pdf'),
                        Button::make('print'),
                        Button::make('reset'),
                        Button::make('reload')
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('DT_RowIndex')
                ->title('#')
                ->searchable(false)
                ->orderable(false)
                ->width(10)
                ->addClass('text-center'),
            Column::make('nama'),
            Column::make('dpt.nama')->title("Nama DPT"),
            Column::make('nik'),
            Column::make('no_wa'),
            Column::make('tps.nomor')->title("TPS Bertugas"),
            Column::make('tps.kelurahan_desa.nama')->title('Kelurahan/Desa'),
            Column::make('partai.nama')->title('Partai'),
            Column::make('suaras_count')->title("Input Suara"),
            Column::make('status'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Saksis_' . date('YmdHis');
    }
}
