<?php

namespace App\DataTables;

use App\Models\KelurahanDesa;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class KelurahanDesasDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
        ->addIndexColumn()
        ->addColumn('action', function ($data) {
            return view('kelurahan-desa.action', compact('data'));
        })
        ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(KelurahanDesa $model): QueryBuilder
    {
        $datas = $model->with('kecamatan.kabupaten_kota');
        if(request()->kecamatan_id){
            $datas = $datas->where('kecamatan_id',request()->kecamatan_id);
        } else if(request()->kabupaten_kota_id){
            $datas = $datas->whereHas('kecamatan',function($w){
                $w->where('kabupaten_kota_id',request()->kabupaten_kota_id);
            });
        } else if(request()->provinsi_id){
            $datas = $datas->whereHas('kecamatan',function($w){
                $w->whereHas('kabupaten_kota',function($w){
                    $w->where('provinsi_id',request()->provinsi_id);
                });
            });
        }
        return $datas;
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()->scrollX(true)
                    ->setTableId('kelurahandesas-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                    ->orderBy(1)
                    ->selectStyleSingle()
                    ->buttons([
                        Button::make('excel'),
                        Button::make('csv'),
                        Button::make('pdf'),
                        Button::make('print'),
                        Button::make('reset'),
                        Button::make('reload')
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('DT_RowIndex')
                ->title('#')
                ->searchable(false)
                ->orderable(false)
                ->width(10)
                ->addClass('text-center'),
            Column::make('nama'),
            Column::make('kecamatan.nama')->title('Kecamatan'),
            Column::make('kecamatan.kabupaten_kota.nama')->title('Kabupaten/Kota'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'KelurahanDesas_' . date('YmdHis');
    }
}
