<?php

namespace App\Http\Controllers;

use App\DataTables\PengumumansDataTable;
use App\Exports\PengumumanJawabsExport;
use App\Helpers\Helper;
use App\Models\Partai;
use App\Models\Pengumuman;
use App\Models\Provinsi;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;
use Str;

class PengumumanController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(PengumumansDataTable $dataTable)
    {
        return $dataTable->render('pengumuman.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {   
        $provinsis = Provinsi::all();
        $partais = Partai::all();
        return view('pengumuman.create',compact('provinsis','partais'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'judul' => 'required',
            'isi' => 'required',

        ]);

        DB::beginTransaction();
        try {
            $pengumuman = Pengumuman::create([
                'user_id' => auth()->user()->id,
                'judul' => $request->judul,
                'isi' => $request->isi,
                'pertanyaan' => $request->pertanyaan != null,
                'aktif' => $request->aktif != null,
                'provinsi_id'=>$request->provinsi_id,
                'kabupaten_kota_id'=>$request->kabupaten_kota_id,
                'kecamatan_id'=>$request->kecamatan_id,
                'kelurahan_desa_id'=>$request->kelurahan_desa_id,
                'user_saksi_id'=>$request->user_saksi_id,
                'partai_id'=>$request->partai_id,
            ]);

            $topic = $_SERVER['SERVER_NAME'];
            if($pengumuman->kabupaten_kota_id){
                $topic = $pengumuman->kabupaten_kota_id;
            }
            if($pengumuman->kecamatan_id){
                $topic = $pengumuman->kabupaten_kota_id."-".$pengumuman->kecamatan_id;
            }
            if($pengumuman->kelurahan_desa_id){
                $topic = $pengumuman->kabupaten_kota_id."-".$pengumuman->kecamatan_id."-".$pengumuman->kelurahan_desa_id;
            }
            if($pengumuman->user_saksi_id){
                $topic = $pengumuman->user_saksi->no_wa;
            }

            if($pengumuman->partai_id){
                $topic .= "-partai-".$pengumuman->partai_id;
            }
            

            if($pengumuman->aktif){
                Helper::sendFcm($pengumuman->judul, $pengumuman->isi, [
                    'id' => (string) $pengumuman->id, 
                    'judul' =>  $pengumuman->judul, 
                    'isi' => $pengumuman->isi,
                    'pertanyaan' => (string) $pengumuman->pertanyaan,
                    'aktif' => (string) $pengumuman->aktif,
                    'diff_human'=>$pengumuman->created_at->diffForHumans()
                ],$topic);
            }

            DB::commit();
            Helper::addUserLog('Membuat pengumuman baru : ' . $pengumuman->judul);
            return redirect()->route('pengumuman.index')->with('success', 'Berhasil membuat pengumuman');
        } catch (\Throwable $th) {
            //throw $th;
            DB::rollBack();
            Log::error($th);
            return redirect()->back()->withInput($request->all())->with('error', 'Gagal membuat pengumuman');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {   
        if (request()->export) {
            $pengumuman = Pengumuman::findOrFail($id);
            $exportClass = new PengumumanJawabsExport($id);
            return Excel::download($exportClass, 'detail_pengumuman_'.Str::slug($pengumuman->judul).'_'. date("YmdHis") . '.xlsx');
        }
        $pengumuman = Pengumuman::with('user_saksi.tps.kelurahan_desa','kabupaten_kota','kecamatan','kelurahan_desa','pengumuman_jawabs.user.tps.kelurahan_desa.kecamatan.kabupaten_kota')->find($id);
        return view('pengumuman.rekap', ['pengumuman' => $pengumuman]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Pengumuman $pengumuman)
    {   
        $provinsis = Provinsi::all();
        $partais = Partai::all();
        return view('pengumuman.edit', ['data' => $pengumuman,'provinsis'=>$provinsis,'partais'=>$partais]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Pengumuman $pengumuman)
    {
        $request->validate([
            'judul' => 'required',
            'isi' => 'required',

        ]);

        DB::beginTransaction();
        try {
            $pengumuman->update([
                'judul' => $request->judul,
                'isi' => $request->isi,
                'pertanyaan' => $request->pertanyaan != null,
                'aktif' => $request->aktif != null,
                'provinsi_id'=>$request->provinsi_id,
                'kabupaten_kota_id'=>$request->kabupaten_kota_id,
                'kecamatan_id'=>$request->kecamatan_id,
                'kelurahan_desa_id'=>$request->kelurahan_desa_id,
                'user_saksi_id'=>$request->user_saksi_id
            ]);

            DB::commit();
            Helper::addUserLog('Mengubah pengumuman : ' . $pengumuman->judul);
            return redirect()->route('pengumuman.index')->with('success', 'Berhasil mengubah pengumuman');
        } catch (\Throwable $th) {
            //throw $th;
            DB::rollBack();
            Log::error($th);
            return redirect()->back()->withInput($request->all())->with('error', 'Gagal mengubah pengumuman');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Pengumuman $pengumuman)
    {
        DB::beginTransaction();
        try {
            $pengumuman->delete();

            DB::commit();
            Helper::addUserLog('Menghapus pengumuman : ' . $pengumuman->judul);
            return redirect()->route('pengumuman.index')->with('success', 'Berhasil menghapus pengumuman');
        } catch (\Throwable $th) {
            //throw $th;
            DB::rollBack();
            Log::error($th);
            return redirect()->back()->with('error', 'Gagal menghapus pengumuman');
        }
    }

    public function sendAgain($id)
    {
        $pengumuman = Pengumuman::find($id);
        if($pengumuman->aktif != 1){
            return redirect()->back()->with('warning', 'Pengumuman tidak aktif');
        }
        $topic = $_SERVER['SERVER_NAME'];
        if($pengumuman->kabupaten_kota_id){
            $topic = $pengumuman->kabupaten_kota_id;
        }
        if($pengumuman->kecamatan_id){
            $topic = $pengumuman->kabupaten_kota_id."-".$pengumuman->kecamatan_id;
        }
        if($pengumuman->kelurahan_desa_id){
            $topic = $pengumuman->kabupaten_kota_id."-".$pengumuman->kecamatan_id."-".$pengumuman->kelurahan_desa_id;
        }
        if($pengumuman->user_saksi_id){
            $topic = $pengumuman->user_saksi->no_wa;
        }
        if($pengumuman->partai_id){
            $topic .= "-partai-".$pengumuman->partai_id;
        }
        $s = Helper::sendFcm($pengumuman->judul, $pengumuman->isi, [
            'id' => (string) $pengumuman->id, 
            'judul' =>  $pengumuman->judul, 
            'isi' => $pengumuman->isi,
            'pertanyaan' => (string) $pengumuman->pertanyaan,
            'aktif' =>  (string) $pengumuman->aktif,
            'diff_human'=>$pengumuman->created_at->diffForHumans()
        ],(string) $topic);
       
        return redirect()->route('pengumuman.index')->with('success', 'Mengirim ulang pengumuman');
    }
}
