<?php

namespace App\DataTables;

use App\Models\UserLog;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class UserLogsDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
        ->addIndexColumn()
        ->editColumn('created_at',function($data){
            return $data->created_at != null ? $data->created_at->format('Y-m-d H:i:s') : '-';
        })
        ->addColumn('data', function ($data) {
            return view('user-log.data', compact('data'));
        })
        ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(UserLog $model): QueryBuilder
    {   
        $datas = $model->select('user_logs.*')->with('user')->with('user.tps','user.tps.kelurahan_desa', 'user.kelurahan_desa', 'user.partai');
        if(request()->tanggal_awal){
            $datas = $datas->whereDate('user_logs.created_at', '>=', request()->tanggal_awal);
        }

        if(request()->tanggal_akhir){
            $datas = $datas->whereDate('user_logs.created_at', '<=', request()->tanggal_akhir);
        }
        return $datas;
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()->scrollX(true)
                    ->setTableId('userlogs-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                    ->orderBy(7)
                    ->selectStyleSingle()
                    ->buttons([
                        Button::make('excel'),
                        Button::make('csv'),
                        Button::make('pdf'),
                        Button::make('print'),
                        Button::make('reset'),
                        Button::make('reload')
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('DT_RowIndex')
                ->title('#')
                ->searchable(false)
                ->orderable(false)
                ->width(10)
                ->addClass('text-center'),
            Column::make('user.nama')->title("Nama"),
            Column::make('user.no_wa')->title("No WA"),
            Column::make('user.tps.nomor')->title("TPS Bertugas"),
            Column::make('user.tps.kelurahan_desa.nama')->title('Kelurahan/Desa'),
            Column::make('message')->title("Keterangan"),
            Column::make('ip_address'),
            Column::make('created_at'),
            Column::computed('data')
                ->exportable(false)
                ->printable(false)
                ->width(100),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'UserLogs_' . date('YmdHis');
    }
}
