<?php

namespace App\DataTables;

use App\Models\User;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class UserKoordinatorsDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addIndexColumn()
            ->addColumn('action', function ($data) {
                return view('user-koordinator.action', compact('data'));
            })
            ->editColumn('status', function ($data) {
                return $data->status == "sudah ada" ? "Sudah Ada" : "Sesuai";
            })
            ->editColumn('multi_tps.nomor', function ($data) {
                return implode(", ", (isset($data->multi_tps)) ? $data->multi_tps->pluck('nomor')->toArray() : []);
            })
            ->orderColumn('multi_tps.nomor', function ($query, $order) {
                $query->orderBy('multi_tps_count', $order);
            })
            ->setRowClass(function ($data) {
                return $data->status_color != 'green' ? 'text-' . $data->status_color . '-500' : '';
            })
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(User $model): QueryBuilder
    {
        $datas = $model->select('users.*')
            ->with('tps', 'tps.kelurahan_desa', 'kelurahan_desa', 'partai', 'multi_tps', 'dpt', 'kecamatan', 'kabupaten_kota')
            ->withCount('multi_tps')
            ->whereIn('role', ['koordinator-kabupaten-kota', 'koordinator-kecamatan', 'koordinator-kelurahan-desa']);

        if (request()->koordinator) {
            $datas = $datas->where('role', 'koordinator-' . request()->koordinator);
        }

        if (auth()->user()->role == "koordinator-kelurahan-desa") {
            $datas = $datas->where('users.kelurahan_desa_id', auth()->user()->kelurahan_desa_id);
        }

        if (auth()->user()->role == "koordinator-kecamatan") {
            $datas = $datas->where('users.kecamatan_id', auth()->user()->kecamatan_id);
        }

        if (auth()->user()->role == "koordinator-kabupaten-kota") {
            $datas = $datas->where('users.kabupaten_kota_id', auth()->user()->kabupaten_kota_id);
        }

        if (request()->kelurahan_desa_id) {
            $datas = $datas->where('users.kelurahan_desa_id', request()->kelurahan_desa_id);
        } else if (request()->kecamatan_id) {
            $datas = $datas->where('users.kecamatan_id', request()->kecamatan_id);
        } else if (request()->kabupaten_kota_id) {
            $datas = $datas->where('users.kabupaten_kota_id', request()->kabupaten_kota_id);
        } else if (request()->provinsi_id) {
            $datas = $datas->where('users.provinsi_id', request()->provinsi_id);
        }

        if (request()->partai_id) {
            $datas = $datas->where('users.partai_id', request()->partai_id);
        }

        $datas = $datas->groupBy('users.id');

        return $datas;
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()->scrollX(true)
            ->setTableId('userkoordinators-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            //->dom('Bfrtip')
            ->orderBy(1)
            ->selectStyleSingle()
            ->buttons([
                Button::make('excel'),
                Button::make('csv'),
                Button::make('pdf'),
                Button::make('print'),
                Button::make('reset'),
                Button::make('reload')
            ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        $columns = [
            Column::make('DT_RowIndex')
                ->title('#')
                ->searchable(false)
                ->orderable(false)
                ->width(10)
                ->addClass('text-center'),
            Column::make('nama'),
            Column::make('dpt.nama')->title("Nama DPT"),
            Column::make('nik'),
            Column::make('no_wa'),

        ];

        if (request()->koordinator == "kelurahan-desa") {
            $columns[] = Column::make('multi_tps.nomor')->title("TPS Bertugas");
            $columns[] = Column::make('tps.kelurahan_desa.nama')->title('Kelurahan/Desa Bertugas');
        }

        // if (request()->koordinator == "kecamatan"){
        //     $columns[] = Column::make('kecamatan.nama')->title('Kecamatan Bertugas');
        // }

        // if (request()->koordinator == "kabupaten-kota"){
        //     $columns[] = Column::make('kabupaten_kota.nama')->title('Kabupaten/Kota Bertugas');
        // }


        $columns = array_merge($columns, [
            request()->koordinator == "kecamatan" ? Column::make('kecamatan.nama')->title('Kecamatan Bertugas') : Column::make('kecamatan.nama')->title('Kecamatan'),
            request()->koordinator == "kabupaten-kota" ? Column::make('kabupaten_kota.nama')->title('Kabupaten/Kota Bertugas') : Column::make('kabupaten_kota.nama')->title('Kabupaten/Kota'),
            Column::make('partai.nama')->title('Partai'),
            Column::make('status'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ]);
        return $columns;
        ;
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'UserKoordinators_' . date('YmdHis');
    }
}
