<?php

namespace App\DataTables;

use App\Models\Tps;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class TpsDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
        ->addIndexColumn()
        ->addColumn('action', function ($data) {
            return view('tps.action', compact('data'));
        })
        ->addColumn('suaras_count', function ($data) {
            return $data->suaras_count > 0 ? "Sudah" : "Belum";
        })
        ->orderColumn('suaras_count', function ($query, $order) {
            return $query->orderBy('suaras_count', $order);
        })
        ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Tps $model): QueryBuilder
    {
        $datas = $model->with('kelurahan_desa.kecamatan.kabupaten_kota')->withCount('suaras');
        if(request()->kelurahan_desa_id){
            $datas = $datas->where('kelurahan_desa_id',request()->kelurahan_desa_id);
        } else if(request()->kecamatan_id){
            $datas = $datas->whereHas('kelurahan_desa',function($w){
                $w->where('kecamatan_id',request()->kecamatan_id);
            });
        } else if(request()->kabupaten_kota_id){
            $datas = $datas->whereHas('kelurahan_desa',function($w){
                $w->whereHas('kecamatan',function($w){
                    $w->where('kabupaten_kota_id',request()->kabupaten_kota_id);
                });
            });
        } else if(request()->provinsi_id){
            $datas = $datas->whereHas('kelurahan_desa',function($w){
                $w->whereHas('kecamatan',function($w){
                    $w->whereHas('kabupaten_kota',function($w){
                        $w->where('provinsi_id',request()->provinsi_id);
                    });
                });
            });
        }

        if(request()->status == "sudah"){
            $datas = $datas->having('suaras_count','=',1);
        }

        if(request()->status == "belum"){
            $datas = $datas->having('suaras_count','=',0);
        }

        return $datas;
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()->scrollX(true)
                    ->setTableId('tps-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                    ->orderBy(1)
                    ->selectStyleSingle()
                    ->buttons([
                        Button::make('excel'),
                        Button::make('csv'),
                        Button::make('pdf'),
                        Button::make('print'),
                        Button::make('reset'),
                        Button::make('reload')
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        $columns = [
            Column::make('DT_RowIndex')
                ->title('#')
                ->searchable(false)
                ->orderable(false)
                ->width(10)
                ->addClass('text-center'),
            Column::make('nomor'),
            Column::make('kelurahan_desa.nama')->title('Kelurahan/Desa'),
            Column::make('kelurahan_desa.kecamatan.nama')->title('Kecamatan'),
            Column::make('kelurahan_desa.kecamatan.kabupaten_kota.nama')->title('Kabupaten/Kota'),
            Column::make('suaras_count')->title("Status"),
            Column::computed('action')
           ->exportable(false)
           ->printable(false)
           ->width(60)
           ->addClass('text-center')
           
        ];
        // if(auth()->user()->role == "admin"){
        //    $columns[] =  Column::computed('action')
        //    ->exportable(false)
        //    ->printable(false)
        //    ->width(60)
        //    ->addClass('text-center');
        // }
        return $columns;
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Tps_' . date('YmdHis');
    }
}
