<?php

namespace App\DataTables;

use App\Models\Calon;
use App\Models\Suara;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class SuarasDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {   
        $calons = Calon::all();
        $elq = (new EloquentDataTable($query))
            ->addIndexColumn()
            ->addColumn('action', function ($data) {
                return view('suara.action', compact('data'));
            })
            ->editColumn('multi_user.nama',function($data){
                return implode(", ",(isset($data->multi_user))?$data->multi_user->pluck('nama')->toArray() : []);
            })
            ->addColumn('lembar_1', function ($data) {
                return view('suara.lembar', ['id'=>$data->lembar_1]);
            })
            ->addColumn('lembar_2', function ($data) {
                return view('suara.lembar', ['id'=>$data->lembar_2]);
            })
            ->addColumn('lembar_3', function ($data) {
                return view('suara.lembar', ['id'=>$data->lembar_3]);
            })
            ->editColumn('created_at',function($data){
                return $data->created_at->format('Y-m-d H:i:s');
            })
            ->editColumn('updated_at',function($data){
                return $data->updated_at != null ? $data->updated_at->format('Y-m-d H:i:s') : 'Belum Diupdate';
            })
            // ->orderColumn('multi_user.nama', function ($query, $order) {
            //     return $query->orderBy('multi_user_count', $order);
            // })
            ->setRowId('id');

        foreach ($calons as $calon) {
            $elq = $elq->filterColumn('jumlah_suara_paslon_'.$calon->nomor, function($query, $keyword)use($calon) {
                $query->havingRaw('jumlah_suara_paslon_'.$calon->nomor.' like ?',["%{$keyword}%"]);
            });
        }

         return  $elq;
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Suara $model): QueryBuilder
    {
        $datas = $model
        ->with('tps.kelurahan_desa.kecamatan.kabupaten_kota','user','multi_user')
        // ->join(DB::raw('tps'),'tps.id','=','suaras.tps_id')
        // ->with(['multi_user'=>function($w){
        //     $w->where('tps_id','=',DB::raw('tps.id'));
        // }])
        ->withCount('multi_user')
        ->with(['tps'=>function($w){
            $w->withCount('dpt');
        }])
        ->with(['suara_details'=>function($w){
            $w->orderBy('calon_id','asc');
        }]);
        $datas = $datas
        ->leftJoin(DB::raw('file_lampirans as l1'),function($j){
            $j->on('l1.suara_id','=','suaras.id')
            ->on('l1.nama','=',DB::raw('"lembar-1"'));
        })
        ->leftJoin(DB::raw('file_lampirans as l2'),function($j){
            $j->on('l2.suara_id','=','suaras.id')
            ->on('l2.nama','=',DB::raw('"lembar-2"'));
        })
        ->leftJoin(DB::raw('file_lampirans as l3'),function($j){
            $j->on('l3.suara_id','=','suaras.id')
            ->on('l3.nama','=',DB::raw('"lembar-3"'));
        });
        if(auth()->user()->role == "koordinator-kelurahan-desa"){
            $datas = $datas->whereHas('tps',function($w){
                $w->where('kelurahan_desa_id',auth()->user()->kelurahan_desa_id);
            });
        }

        if(auth()->user()->role == "koordinator-kecamatan"){
            $datas = $datas->whereHas('tps.kelurahan_desa',function($w){
                $w->where('kecamatan_id',auth()->user()->kecamatan_id);
            });
        }

        if(auth()->user()->role == "koordinator-kabupaten-kota"){
            $datas = $datas->whereHas('tps.kelurahan_desa.kecamatan',function($w){
                $w->where('kabupaten_kota_id',auth()->user()->kabupaten_kota_id);
            });
        }
    
        if (request()->kelurahan_desa_id) {
            $datas = $datas->whereHas('tps',function($w){
                $w->where('kelurahan_desa_id', request()->kelurahan_desa_id);
            });
        } else if (request()->kecamatan_id) {
            $datas = $datas->whereHas('tps',function($w){
                $w->whereHas('kelurahan_desa', function ($w) {
                    $w->where('kecamatan_id', request()->kecamatan_id);
                });
            });
        } else if (request()->kabupaten_kota_id) {
            $datas = $datas->whereHas('tps',function($w){
                $w->whereHas('kelurahan_desa', function ($w) {
                    $w->whereHas('kecamatan', function ($w) {
                        $w->where('kabupaten_kota_id', request()->kabupaten_kota_id);
                    });
                });
            });
        } else if (request()->provinsi_id) {
            $datas = $datas->whereHas('tps',function($w){
                $w->whereHas('kelurahan_desa', function ($w) {
                    $w->whereHas('kecamatan', function ($w) {
                        $w->whereHas('kabupaten_kota', function ($w) {
                            $w->where('provinsi_id', request()->provinsi_id);
                        });
                    });
                });
            });
        }

        if (request()->partai_id) {
            $datas = $datas->whereHas('multi_user',function($w){
                $w->where('partai_id', request()->partai_id);
            });
        }

        if(request()->tanggal_awal){
            $datas = $datas->whereDate('suaras.created_at', '>=', request()->tanggal_awal);
        }

        if(request()->tanggal_akhir){
            $datas = $datas->whereDate('suaras.created_at', '<=', request()->tanggal_akhir);
        }
       
        $calons = Calon::all();
        $selects = ['suaras.*',DB::raw('l1.id as lembar_1'),DB::raw('l2.id as lembar_2'),DB::raw('l3.id as lembar_3')];
        foreach ($calons as $calon) {
            $datas = $datas->leftJoin('suara_details as paslon_'.$calon->nomor,function($w)use($calon){
                $w->on('paslon_'.$calon->nomor.'.suara_id','=','suaras.id')
                ->on('paslon_'.$calon->nomor.'.calon_id','=',DB::raw($calon->id));
            });
            $selects[] = DB::raw('SUM(paslon_'.$calon->nomor.'.jumlah_suara) as jumlah_suara_paslon_'.$calon->nomor);
            
        }
        $datas = $datas->select($selects);
        $datas = $datas->groupBy('suaras.id');

        return $datas;
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()->scrollX(true)->scrollX(true)
                    ->setTableId('suaras-table')
                    ->columns($this->getColumns())
                    ->minifiedAjax()
                    //->dom('Bfrtip')
                    ->orderBy(1)
                    ->selectStyleSingle()
                    ->buttons([
                        Button::make('excel'),
                        Button::make('csv'),
                        Button::make('pdf'),
                        Button::make('print'),
                        Button::make('reset'),
                        Button::make('reload')
                    ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {   
        $columns = [
            Column::make('DT_RowIndex')
                ->title('#')
                ->searchable(false)
                ->orderable(false)
                ->width(10)
                ->addClass('text-center'),
            Column::make('created_at')->name('suaras.created_at')->title('Dibuat')->addClass('whitespace-nowrap'),
            Column::make('updated_at')->name('suaras.updated_at')->title('Diupdate')->addClass('whitespace-nowrap'),
            Column::make('tps.kelurahan_desa.kecamatan.kabupaten_kota.nama')->title('Kabupaten/Kota')->addClass('whitespace-nowrap'),
            Column::make('tps.kelurahan_desa.kecamatan.nama')->title("Kecamatan")->addClass('whitespace-nowrap'),
            Column::make('tps.kelurahan_desa.nama')->title('Kelurahan/Desa')->addClass('whitespace-nowrap'),
            Column::make('tps.nomor')->title('TPS'),
        ];

        $calons = Calon::all();
        foreach ($calons as $calon) {
            $columns[] = Column::make('jumlah_suara_paslon_'.$calon->nomor)->title("Paslon ".$calon->nomor);
        }
        $columns[] = Column::make('multi_user.nama')->title("Saksi Bertugas");
        $columns[] =  Column::make('lembar_1');
        $columns[] =  Column::make('lembar_2');
        $columns[] =  Column::make('lembar_3');
        $columns[] = Column::computed('action')
        ->exportable(false)
        ->printable(false)
        ->width(60)
        ->addClass('text-center');
        return $columns;
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Suaras_' . date('YmdHis');
    }
}
