<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Produk;
use Illuminate\Support\Facades\DB;
use App\Helpers\Helper;

class ProdukController extends Controller
{

    protected $routeName = 'produk';
    protected $viewName = 'produk';
    protected $menu = 'Master Data';
    protected $title = 'Produk';

    protected $aksesMenu = 'Produk';
    
    public function index()
    {
        if(!Helper::cek_akses($this->aksesMenu)){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'ubah'=>Helper::cek_akses($this->aksesMenu,'Ubah'),
            'hapus'=>Helper::cek_akses($this->aksesMenu,'Hapus')
        ];

        $datas = Produk::all();
        
        return view($this->viewName.'.index', compact('settings', 'datas'));
    }
    
    public function create()
    {
        if(!Helper::cek_akses($this->aksesMenu, 'Tambah')){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'action'=> route($this->routeName.'.store'),
        ];
        
        return view($this->viewName.'.form', compact('settings'));
    }
    
    public function store(Request $request)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Tambah')){
            return abort(404);
        }

        $request->validate([
            'nama_produk'=>'required',
            'satuan'=>'required',
            'harga_jual'=>'required',
            'harga_beli'=>'required',
            'stok'=>'required',
        ]);
        
        $request->harga_jual = Helper::onlyNumber($request->harga_jual);

        DB::beginTransaction();

        try {
            $produk = Produk::create([
               'nama_produk'=>$request->nama_produk,
               'satuan'=>$request->satuan,
               'harga_jual'=>Helper::onlyNumber($request->harga_jual),
               'harga_beli'=>Helper::onlyNumber($request->harga_beli),
               'stok_awal'=>$request->stok,
               'stok'=>$request->stok,
            ]);

            DB::commit();

            Helper::addUserLog('Menambah data ' .$this->aksesMenu . ' : ' . $produk->nama_produk,$produk->toArray());
            
            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Menambah Data ' .$this->aksesMenu. ' : '.$produk->nama_produk]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : '.$e->getMessage()])->withInput($request->all());
        }
    }

    public function show($id)
    {
        //
    }
    
    public function edit(produk $produk)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Ubah')){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'action'=>route($this->routeName.'.update',$produk->id),
        ];

        return view($this->viewName.'.form',compact('settings'),['data'=>$produk]);
    }

    public function update(Request $request, produk $produk)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Ubah')){
            return abort(404);
        }
        
        $request->validate([
            'nama_produk'=>'required',
            'satuan'=>'required',
            'harga_jual'=>'required',
            'harga_beli'=>'required',
        ]);

        $request->harga_jual = Helper::onlyNumber($request->harga_jual);
        
        DB::beginTransaction();
        try {
            $old = $produk->toArray();
            $produk->update([
                'nama_produk'=>$request->nama_produk,
                'satuan'=>$request->satuan,
                'harga_jual'=>Helper::onlyNumber($request->harga_jual),
                'harga_beli'=>Helper::onlyNumber($request->harga_beli),
            ]);
            DB::commit();

            Helper::addUserLog('Menambah data ' .$this->aksesMenu . ' : ' . $produk->nama_produk,[
                'old' => $old,
                'update' => $produk->toArray()
            ]);

            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Mengubah Data ' .$this->aksesMenu. ' : '.$produk->nama_produk]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Mengubah Data ' .$this->aksesMenu. ' : '.$e->getMessage()])->withInput($request->all());
        }
    }
    
    public function destroy(produk $produk)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Hapus')){
            return abort(404);
        }

        DB::beginTransaction();
        try {
            Helper::addUserLog('Menghapus data ' .$this->aksesMenu . ' : ' . $produk->nama_produk,$produk->toArray());

            $produk->delete();
        
            DB::commit();
            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Menghapus Data produk : '.$produk->nama_produk]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Menghapus Data produk '.$produk->nama_produk.' : '.$e->getMessage()]);
        }
    }
}
