<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PembayaranPenjualan;
use App\Models\DetailPenjualan;
use App\Models\Penjualan;
use App\Models\Produk;
use App\Models\JurnalUmum;
use App\Models\Akun;
use App\Models\Pelanggan;
use App\Helpers\Helper;
use Illuminate\Support\Facades\DB;

use PDF;

class PenjualanController extends Controller
{

    protected $routeName = 'penjualan';
    protected $viewName = 'penjualan';
    protected $menu = 'Master Data';
    protected $title = 'Penjualan';

    protected $aksesMenu = 'Penjualan';
    
    public function index()
    {
        if(!Helper::cek_akses($this->aksesMenu)){
            return abort(404);
        }

        // instantiate and use the dompdf class
        // $dompdf = new Dompdf();

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'ubah'=>Helper::cek_akses($this->aksesMenu,'Ubah'),
            'hapus'=>Helper::cek_akses($this->aksesMenu,'Hapus')
        ];

        
        $datas = Penjualan::orderBy('created_at', 'desc')->get();

        return view($this->viewName.'.index', compact('settings', 'datas'));
    }
    
    public function create()
    {
        if(!Helper::cek_akses($this->aksesMenu, 'Tambah')){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'action'=> route($this->routeName.'.store'),
        ];

        $produks = Produk::all();
        $pelanggans = Pelanggan::all();
        $akuns = Akun::all();
        
        return view($this->viewName.'.form', compact('settings', 'produks', 'pelanggans', 'akuns'));
    }
    
    public function store(Request $request)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Tambah')){
            return abort(404);
        }

        $request->validate([
            'pelanggan_id'=>'required',
            'no_transaksi'=>'required',
            'tgl_transaksi'=>'required',
            'tgl_jatuh_tempo'=>'required',
        ]);
        DB::beginTransaction();
        try {
            if(count($request->produk_id) <= 0)
                return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Masukkan Setidaknya Satu Produk'])->withInput($request->all());

            $penjualan = Penjualan::create([
               'pelanggan_id'=>$request->pelanggan_id,
               'tgl_transaksi'=>$request->tgl_transaksi,
               'tgl_jatuh_tempo'=>$request->tgl_jatuh_tempo,
               'no_transaksi'=> $request->no_transaksi,
               'catatan'=>$request->catatan,
            ]);

            $total = 0;
            $hpp = 0;
            $totalDiskon = 0;
            $totalPajak = 0;
            $totalPendapatan = 0;
            $totalPiutang = 0;
            $totalAwal = 0;

            for ($i=0; $i < count($request->produk_id); $i++) { 
                $produk = Produk::find($request->produk_id[$i]);

                if(!$produk)
                    return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Produk Dengan Id ' . $request->produk_id[$i] . 'Tidak Ditemukan'])->withInput($request->all());

                if($produk->stok < $request->qty[$i])
                    return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Produk Dengan Nama ' . $produk->nama_produk . ' Kekurangan Stok..'])->withInput($request->all());

                $diskon = 0;
                if ($request->tipe_diskon == 'rp') {
                    $diskon = $request->diskon[$i] ? Helper::onlyNumber($request->diskon[$i]) : 0;
                }elseif($request->tipe_diskon == 'persen'){
                    $harga = Helper::onlyNumber($request->harga_jual[$i]) * Helper::onlyFloat($request->qty[$i]);
                    $diskon = $request->diskon[$i] ? $harga * Helper::onlyNumber($request->diskon[$i]) / 100 : 0;
                }

                $subtotal = (Helper::onlyNumber($request->harga_jual[$i]) * Helper::onlyFloat($request->qty[$i])) - $diskon;
                
                $pajak = 0;
                if($request->pajak[$i]){
                    $pajak = ($subtotal * Helper::onlyNumber($request->pajak[$i]) / 100 );
                    $subtotal =  $subtotal + $pajak;
                }else if($request->pajak_lain[$i]) {
                    $pajak = ($subtotal * Helper::onlyNumber($request->pajak_lain[$i]) / 100 );
                    $subtotal=  $subtotal + $pajak;
                }
                
                $hpp += $produk->harga_beli * $request->qty[$i];
                $total += $subtotal;
                $totalDiskon += $diskon;
                $totalPajak += $pajak;

                $totalPiutang += (Helper::onlyNumber($request->harga_jual[$i]) * Helper::onlyFloat($request->qty[$i])) - $diskon + $pajak;
                $totalPendapatan += (Helper::onlyNumber($request->harga_jual[$i]) * Helper::onlyFloat($request->qty[$i]));
                
                DetailPenjualan::create([
                    'penjualan_id' => $penjualan->id,
                    'produk_id' => $request->produk_id[$i],
                    'qty' => $request->qty[$i],
                    'harga' => Helper::onlyNumber($request->harga_jual[$i]),
                    'diskon' => $diskon,
                    'pajak' => ($request->pajak[$i]) ? Helper::onlyNumber($request->pajak[$i]) : null,
                    'pajak_lain' => ($request->pajak_lain[$i]) ? Helper::onlyNumber($request->pajak_lain[$i]) : null,
                ]);

                $produk->update([
                    'stok' => $produk->stok - $request->qty[$i]
                ]);
                $totalAwal += Helper::onlyFloat($request->qty[$i]) * Helper::onlyNumber($request->harga_jual[$i]);

            }

            if(!$total)
                return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Error, Total Gagal Dihitung '])->withInput($request->all());

            $status = 'pending';

            $pemotongan = 0;
            if ($request->pemotongan && $request->akun_pemotongan) {
                if ($request->tipe_pemotongan == 'rp') {
                    $pemotongan = Helper::onlyNumber($request->pemotongan);
                }else if($request->tipe_pemotongan == 'persen'){
                    $pemotongan = $totalAwal * ($request->pemotongan / 100);
                }
            }
            $penjualan->update([
                'total' => $total - $pemotongan,
                'status' => $status,
                'pemotongan_akun_id' => $request->akun_pemotongan ?? NULL,
            ]);

            if ($request->pemotongan && $request->akun_pemotongan) {
                $penjualan->update([
                    'pemotongan' => $pemotongan,
                ]);
            }

            $akunDiskon = Akun::where('nama_akun', 'Diskon Penjualan')->first();
            $akunPajak = Akun::where('nama_akun', 'PPN Keluaran')->first();
            $akunPiutang = Akun::where('nama_akun', 'Akun Piutang')->first();
            $akunBiayaPenjualan = Akun::where('nama_akun', 'Biaya Penjualan (HPP)')->first();
            $akunPendapatan = Akun::where('nama_akun', 'Akun Pendapatan')->first();
            $akunPersediaan = Akun::where('nama_akun', 'Akun Persediaan')->first();

            if ($request->pemotongan && $request->akun_pemotongan) {
                $akunPemotongan = Akun::find($request->akun_pemotongan);

                // Debet ke Akun Pemotongan
                JurnalUmum::create([
                    'akun_id' => $akunPemotongan->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'penjualan_id' => $penjualan->id,
                    'debet' => $pemotongan,
                    'description' => 'Pemotongan Produk'
                ]);
            }

            if ($totalDiskon > 0) {
                // Debet ke Akun Diskon
                JurnalUmum::create([
                    'akun_id' => $akunDiskon->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'penjualan_id' => $penjualan->id,
                    'debet' => $totalDiskon,
                ]);
            }

            // Debet ke Akun Piutang
            JurnalUmum::create([
                'akun_id' => $akunPiutang->id,
                'tanggal' => $request->tgl_transaksi,
                'pelanggan_id' => $request->pelanggan_id,
                'penjualan_id' => $penjualan->id,
                'debet' => $totalPiutang - $pemotongan,
            ]);

            // Debet ke Akun Biaya Penjualan
            JurnalUmum::create([
                'akun_id' => $akunBiayaPenjualan->id,
                'tanggal' => $request->tgl_transaksi,
                'pelanggan_id' => $request->pelanggan_id,
                'penjualan_id' => $penjualan->id,
                'debet' => $hpp,
            ]);

            // Kredit ke Akun Pendapatan
            JurnalUmum::create([
                'akun_id' => $akunPendapatan->id,
                'tanggal' => $request->tgl_transaksi,
                'pelanggan_id' => $request->pelanggan_id,
                'penjualan_id' => $penjualan->id,
                'kredit' => $totalPendapatan,
            ]);

            if ($totalPajak > 0) {
                // Kredit ke Akun PPN
                JurnalUmum::create([
                    'akun_id' => $akunPajak->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'penjualan_id' => $penjualan->id,
                    'kredit' => $totalPajak,
                ]);
            }

            // Kredit ke Akun Persediaan
            JurnalUmum::create([
                'akun_id' => $akunPersediaan->id,
                'tanggal' => $request->tgl_transaksi,
                'pelanggan_id' => $request->pelanggan_id,
                'penjualan_id' => $penjualan->id,
                'kredit' => $hpp,
            ]);

            DB::commit();

            Helper::addUserLog('Menambah data ' .$this->aksesMenu . ' : ' . $penjualan->no_transaksi,$penjualan->toArray());
            
            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Menambah Data ' .$this->aksesMenu. ' : '.$penjualan->no_transaksi]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : '.$e->getMessage()])->withInput($request->all());
        }
    }

    public function show(Penjualan $penjualan)
    {
        if(!Helper::cek_akses($this->aksesMenu)){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'action'=>route($this->routeName.'.bayar',$penjualan->id),
        ];

        $produks = Produk::all();
        $pelanggans = Pelanggan::all();

        $akuns = Akun::all();

        return view($this->viewName.'.show',compact('settings', 'produks', 'akuns', 'pelanggans'),['data'=>$penjualan]);
    }
    
    public function edit(Penjualan $penjualan)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Ubah')){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'action'=>route($this->routeName.'.update',$penjualan->id),
        ];

        $produks = Produk::all();
        $pelanggans = Pelanggan::all();
        $akuns = Akun::all();

        return view($this->viewName.'.form',compact('settings', 'produks', 'pelanggans', 'akuns'),['data'=>$penjualan]);
    }

    public function update(Request $request, penjualan $penjualan)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Ubah')){
            return abort(404);
        }
        
        $request->validate([
            'pelanggan_id'=>'required',
            'no_transaksi'=>'required',
            'tgl_transaksi'=>'required',
            'tgl_jatuh_tempo'=>'required',
        ]);
        
        DB::beginTransaction();
        try {
            if(count($request->produk_id) <= 0)
                return redirect()->back()->with(['error'=>'Gagal Mengubah Data ' .$this->aksesMenu. ' : Masukkan Setidaknya Satu Produk'])->withInput($request->all());

            $old = $penjualan->toArray();
            
            $penjualan->update([
                'pelanggan_id'=>$request->pelanggan_id,
                'no_transaksi'=>$request->no_transaksi,
                'tgl_transaksi'=>$request->tgl_transaksi,
                'tgl_jatuh_tempo'=>$request->tgl_jatuh_tempo,
                'catatan'=>$request->catatan,
            ]);
 
            $total = 0;
            $hpp = 0;
            $totalDiskon = 0;
            $totalPajak = 0;
            $totalPendapatan = 0;
            $totalPiutang = 0;

            // Kembalikan Stok, Kemudian Hapus Detail
            for ($i=0; $i < count($penjualan->detail); $i++) {
                $produk = Produk::find($request->produk_id[$i]);
                $produk->update([
                    'stok' => $produk->stok + $penjualan->detail[$i]->qty
                ]);
            }

            $penjualan->detail()->delete();
            for ($i=0; $i < count($request->produk_id); $i++) { 
                $produk = Produk::find($request->produk_id[$i]);

                if(!$produk)
                    return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Produk Dengan Id ' . $request->produk_id[$i] . 'Tidak Ditemukan'])->withInput($request->all());

                if($produk->stok < $request->qty[$i])
                    return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Produk Dengan Nama ' . $produk->nama_produk . ' Kekurangan Stok..'])->withInput($request->all());

                $diskon = 0;
                if ($request->tipe_diskon == 'rp') {
                    $diskon = $request->diskon[$i] ? Helper::onlyNumber($request->diskon[$i]) : 0;
                }elseif($request->tipe_diskon == 'persen'){
                    $harga = Helper::onlyNumber($request->harga_jual[$i]) * Helper::onlyFloat($request->qty[$i]);
                    $diskon = $request->diskon[$i] ? $harga * Helper::onlyNumber($request->diskon[$i]) / 100 : 0;
                }

                $subtotal = (Helper::onlyNumber($request->harga_jual[$i]) * Helper::onlyFloat($request->qty[$i])) - $diskon;

                $pajak = 0;
                if($request->pajak[$i]){
                    $pajak = ($subtotal * Helper::onlyNumber($request->pajak[$i]) / 100 );
                    $subtotal =  $subtotal + $pajak;
                }else if($request->pajak_lain[$i]) {
                    $pajak = ($subtotal * Helper::onlyNumber($request->pajak_lain[$i]) / 100 );
                    $subtotal=  $subtotal + $pajak;
                }

                $hpp += $produk->harga_beli * $request->qty[$i];
                $total += $subtotal;
                $totalDiskon += $diskon;
                $totalPajak += $pajak;

                $totalPiutang += (Helper::onlyNumber($request->harga_jual[$i]) * Helper::onlyFloat($request->qty[$i])) - $diskon + $pajak;
                $totalPendapatan += (Helper::onlyNumber($request->harga_jual[$i]) * Helper::onlyFloat($request->qty[$i]));
                
                $detail = DetailPenjualan::create([
                    'penjualan_id' => $penjualan->id,
                    'produk_id' => $request->produk_id[$i],
                    'qty' => $request->qty[$i],
                    'harga' => Helper::onlyNumber($request->harga_jual[$i]),
                    'diskon' => $diskon,
                    'pajak' => ($request->pajak[$i]) ? Helper::onlyNumber($request->pajak[$i]) : null,
                    'pajak_lain' => ($request->pajak_lain[$i]) ? Helper::onlyNumber($request->pajak_lain[$i]) : null,
                ]);

                $produk->update([
                    'stok' => $produk->stok - $request->qty[$i]
                ]);

            }

            if(!$total)
                return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Error, Total Gagal Dihitung '])->withInput($request->all());

            $pemotongan = 0;
            if ($request->pemotongan && $request->akun_pemotongan) {
                if ($request->tipe_pemotongan == 'rp') {
                    $pemotongan = Helper::onlyNumber($request->pemotongan);
                }else if($request->tipe_pemotongan == 'persen'){
                    $pemotongan = $total * Helper::onlyNumber($request->pemotongan) / 100;
                }
            }

            $penjualan->update([
                'total' => $total - $pemotongan,
                'pemotongan_akun_id' => $request->akun_pemotongan ?? NULL,
            ]);

            if ($request->pemotongan && $request->akun_pemotongan) {
                $penjualan->update([
                    'pemotongan' => $pemotongan,
                ]);
            }

            $akunDiskon = Akun::where('nama_akun', 'Diskon Penjualan')->first();
            $akunPajak = Akun::where('nama_akun', 'PPN Keluaran')->first();
            $akunPiutang = Akun::where('nama_akun', 'Akun Piutang')->first();
            $akunBiayaPenjualan = Akun::where('nama_akun', 'Biaya Penjualan (HPP)')->first();
            $akunPendapatan = Akun::where('nama_akun', 'Akun Pendapatan')->first();
            $akunPersediaan = Akun::where('nama_akun', 'Akun Persediaan')->first();

            $penjualan->JurnalUmum()->delete();

            if ($request->pemotongan && $request->akun_pemotongan) {
                $akunPemotongan = Akun::find($request->akun_pemotongan);

                // Debet ke Akun Pemotongan
                JurnalUmum::create([
                    'akun_id' => $akunPemotongan->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'penjualan_id' => $penjualan->id,
                    'debet' => $pemotongan,
                    'description' => 'Pemotongan Produk'
                ]);
            }

            if ($totalDiskon > 0) {
                // Debet ke Akun Diskon
                JurnalUmum::create([
                    'akun_id' => $akunDiskon->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'penjualan_id' => $penjualan->id,
                    'debet' => $totalDiskon,
                ]);
            }
            
            // Debet ke Akun Piutang
            JurnalUmum::create([
                'akun_id' => $akunPiutang->id,
                'tanggal' => $request->tgl_transaksi,
                'pelanggan_id' => $request->pelanggan_id,
                'penjualan_id' => $penjualan->id,
                'debet' => $totalPiutang - $pemotongan,
            ]);

            // Debet ke Akun Biaya Penjualan
            JurnalUmum::create([
                'akun_id' => $akunBiayaPenjualan->id,
                'tanggal' => $request->tgl_transaksi,
                'pelanggan_id' => $request->pelanggan_id,
                'penjualan_id' => $penjualan->id,
                'debet' => $hpp,
            ]);

            // Kredit ke Akun Pendapatan
            JurnalUmum::create([
                'akun_id' => $akunPendapatan->id,
                'tanggal' => $request->tgl_transaksi,
                'supplier_id' => $request->supplier_id,
                'penjualan_id' => $penjualan->id,
                'kredit' => $totalPendapatan,
            ]);

            if ($totalPajak > 0) {
                // Kredit ke Akun PPN
                JurnalUmum::create([
                    'akun_id' => $akunPajak->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'penjualan_id' => $penjualan->id,
                    'kredit' => $totalPajak,
                ]);
            }

            // Kredit ke Akun Persediaan
            JurnalUmum::create([
                'akun_id' => $akunPersediaan->id,
                'tanggal' => $request->tgl_transaksi,
                'supplier_id' => $request->supplier_id,
                'penjualan_id' => $penjualan->id,
                'kredit' => $hpp,
            ]);

            DB::commit();

            Helper::addUserLog('Menambah data ' .$this->aksesMenu . ' : ' . $penjualan->no_transaksi,[
                'old' => $old,
                'update' => $penjualan->toArray()
            ]);

            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Mengubah Data ' .$this->aksesMenu. ' : '.$penjualan->no_transaksi]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Mengubah Data ' .$this->aksesMenu. ' : '.$e->getMessage()])->withInput($request->all());
        }
    }

    public function bayar(Request $request, $id)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Tambah')){
            return abort(404);
        }
        
        $request->validate([
            'akun_id'=>'required',
            'tgl_bayar'=>'required',
            'jumlah_bayar'=>'required',
        ]);
        
        DB::beginTransaction();
        try {
            $penjualan = Penjualan::find($id);
            
            $pembayaran_penjualan = PembayaranPenjualan::create([
                'akun_id'=>$request->akun_id,
                'penjualan_id' => $penjualan->id,
                'tgl_bayar' => $request->tgl_bayar,
                'jumlah_bayar' => Helper::onlyNumber($request->jumlah_bayar),
            ]);

            if($penjualan->sisaTagihan > 0){
                $status = 'pending';
            }elseif($penjualan->sisaTagihan <= 0){
                $status = 'paid';
            }

            $penjualan->update([
                'status' => $status,
            ]);

            $akunPiutang = Akun::where('nama_akun', 'Akun Piutang')->first();
            $akunKas = Akun::find($request->akun_id);
            
            // Debet ke Akun Kas
            JurnalUmum::create([
                'akun_id' => $akunKas->id,
                'tanggal' => $request->tgl_bayar,
                'pelanggan_id' => $penjualan->pelanggan_id,
                'pembayaran_penjualan_id' => $pembayaran_penjualan->id,
                'debet' => Helper::onlyNumber($request->jumlah_bayar),
            ]);

            // Kredit ke Akun Piutang
            JurnalUmum::create([
                'akun_id' => $akunPiutang->id,
                'tanggal' => $request->tgl_bayar,
                'pelanggan_id' => $penjualan->pelanggan_id,
                'pembayaran_penjualan_id' => $pembayaran_penjualan->id,
                'kredit' => Helper::onlyNumber($request->jumlah_bayar),
            ]);

            DB::commit();

            Helper::addUserLog('Menambah data ' .$this->aksesMenu . ' Pembayaran : ' . $penjualan->no_transaksi,$penjualan->toArray());

            return redirect(route($this->routeName.'.show',$id))->with(['success'=>'Berhasil Menambah Data ' .$this->aksesMenu. ' Pembayaran: '.$penjualan->no_transaksi]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Mengubah Data ' .$this->aksesMenu. ' Pembayaran : '.$e->getMessage()])->withInput($request->all());
        }
    }
    
    public function update_detail(Request $request)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Ubah')){
            return abort(404);
        }

        DB::beginTransaction();
        try {
            
            $pembayaran_penjualan = PembayaranPenjualan::find($request->id);
            
            $pembayaran_penjualan->update([
                'jumlah_bayar' => $request->harga
            ]);

            
            $akunPiutang = Akun::where('nama_akun', 'Akun Piutang')->first();
            $akunKas = Akun::where('nama_akun', 'Kas')->first();

            $jurnalUmumKas = JurnalUmum::where('pembayaran_penjualan_id', $pembayaran_penjualan->id)->where('akun_id', $akunKas->id)->first();

            $jurnalUmumPiutang = JurnalUmum::where('pembayaran_penjualan_id', $pembayaran_penjualan->id)->where('akun_id', $akunPiutang->id)->first();
            
            // Debet ke Akun Kas
            $jurnalUmumKas->update([
                'debet' => Helper::onlyNumber($request->harga),
            ]);
            
            // Kredit ke Akun Piutang
            $jurnalUmumPiutang->update([
                'kredit' => Helper::onlyNumber($request->harga),
            ]);

            DB::commit();
            $data = [
                'status' => 200,
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            $data = [
                'status' => 500,
                'message' => 'Tidak Dapat diubah' . $e->getMessage(),
                'data' => null,
            ];
        }
        return response()->json($data,$data['status']);
    }

    public function destroy(penjualan $penjualan)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Hapus')){
            return abort(404);
        }

        DB::beginTransaction();
        try {
            Helper::addUserLog('Menghapus data ' .$this->aksesMenu . ' : ' . $penjualan->no_transaksi,$penjualan->toArray());

            foreach ($penjualan->pembayaran as $pembayaran) {
                $pembayaran->delete();
            }
            $penjualan->delete();
        
            DB::commit();
            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Menghapus Data penjualan : '.$penjualan->no_transaksi]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Menghapus Data penjualan '.$penjualan->no_transaksi.' : '.$e->getMessage()]);
        }
    }

    public function print(penjualan $penjualan){
        if(!Helper::cek_akses($this->aksesMenu)){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
        ];

        $produks = Produk::all();
        $pelanggans = Pelanggan::all();

        return PDF::loadview($this->viewName.'.print',compact('settings', 'produks', 'pelanggans'),['data'=>$penjualan])->setPaper('A4', 'portrait')->stream();

        return view($this->viewName.'.print',compact('settings', 'produks', 'pelanggans'),['data'=>$penjualan]);
    }
}
