<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PembayaranPembelian;
use App\Models\DetailPembelian;
use App\Models\Pembelian;
use App\Models\Produk;
use App\Models\Akun;
use App\Models\JurnalUmum;
use App\Models\Supplier;
use App\Helpers\Helper;
use Illuminate\Support\Facades\DB;

class PembelianController extends Controller
{

    protected $routeName = 'pembelian';
    protected $viewName = 'pembelian';
    protected $menu = 'Master Data';
    protected $title = 'Pembelian';

    protected $aksesMenu = 'Pembelian';
    
    public function index()
    {
        if(!Helper::cek_akses($this->aksesMenu)){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'ubah'=>Helper::cek_akses($this->aksesMenu,'Ubah'),
            'hapus'=>Helper::cek_akses($this->aksesMenu,'Hapus')
        ];

        $datas = Pembelian::orderBy('created_at', 'desc')->get();
        
        return view($this->viewName.'.index', compact('settings', 'datas'));
    }
    
    public function create()
    {
        if(!Helper::cek_akses($this->aksesMenu, 'Tambah')){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'action'=> route($this->routeName.'.store'),
        ];

        $produks = Produk::all();
        $suppliers = Supplier::all();
        $akuns = Akun::all();

        return view($this->viewName.'.form', compact('settings', 'produks', 'suppliers', 'akuns'));
    }
    
    public function store(Request $request)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Tambah')){
            return abort(404);
        }

        $request->validate([
            'supplier_id'=>'required',
            'tgl_transaksi'=>'required',
            'tgl_jatuh_tempo'=>'required',
        ]);

        DB::beginTransaction();
        try {
            if(count($request->produk_id) <= 0)
            return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Masukkan Setidaknya Satu Produk'])->withInput($request->all());

            $pembelian = Pembelian::create([
               'supplier_id'=>$request->supplier_id,
               'tgl_transaksi'=>$request->tgl_transaksi,
               'tgl_jatuh_tempo'=>$request->tgl_jatuh_tempo,
               'no_transaksi'=> Helper::generateNopembelian(),
               'catatan'=>$request->catatan,
            ]);

            $total = 0;
            $totalDiskon = 0;
            $totalPajak = 0;
            for ($i=0; $i < count($request->produk_id); $i++) { 
                $produk = Produk::find($request->produk_id[$i]);

                if(!$produk)
                    return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Produk Dengan Id ' . $request->produk_id[$i] . 'Tidak Ditemukan'])->withInput($request->all());
                    
                $diskon = 0;
                if ($request->tipe_diskon == 'rp') {
                    $diskon = $request->diskon[$i] ? Helper::onlyNumber($request->diskon[$i]) : 0;
                }elseif($request->tipe_diskon == 'persen'){
                    $harga = Helper::onlyNumber($request->harga_beli[$i]) * Helper::onlyFloat($request->qty[$i]);
                    $diskon = $request->diskon[$i] ? $harga * Helper::onlyNumber($request->diskon[$i]) / 100 : 0;
                }

                $subtotal = (Helper::onlyNumber($request->harga_beli[$i]) * Helper::onlyFloat($request->qty[$i])) - $diskon;

                $pajak = 0;
                if($request->pajak[$i]){
                    $pajak = ($subtotal * Helper::onlyNumber($request->pajak[$i]) / 100 );
                    $subtotal =  $subtotal + $pajak;
                }else if($request->pajak_lain[$i]) {
                    $pajak = ($subtotal * Helper::onlyNumber($request->pajak_lain[$i]) / 100 );
                    $subtotal=  $subtotal + $pajak;
                }

                $total += $subtotal;
                $totalDiskon += $diskon;
                $totalPajak += $pajak;
                
                $detail = DetailPembelian::create([
                    'pembelian_id' => $pembelian->id,
                    'produk_id' => $request->produk_id[$i],
                    'diskon' => $diskon,
                    'qty' => $request->qty[$i],
                    'harga' => Helper::onlyNumber($request->harga_beli[$i]),
                    'pajak' => ($request->pajak[$i]) ? Helper::onlyNumber($request->pajak[$i]) : null,
                    'pajak_lain' => ($request->pajak_lain[$i]) ? Helper::onlyNumber($request->pajak_lain[$i]) : null,
                ]);

                $produk->update([
                    'stok' => $produk->stok + $request->qty[$i]
                ]);

            }

            if(!$total)
                return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Error, Total Gagal Dihitung '])->withInput($request->all());
                
            $status = 'pending';

            $pemotongan = 0;
            if ($request->pemotongan && $request->akun_pemotongan) {
                if ($request->tipe_pemotongan == 'rp') {
                    $pemotongan = Helper::onlyNumber($request->pemotongan);
                }else if($request->tipe_pemotongan == 'persen'){
                    $pemotongan = $total * Helper::onlyNumber($request->pemotongan) / 100;
                }
            }

            $pembelian->update([
                'total' => $total - $pemotongan,
                'status' => $status,
                'pemotongan_akun_id' => $request->akun_pemotongan ?? NULL,
            ]);

            if ($request->pemotongan && $request->akun_pemotongan) {
                $pembelian->update([
                    'pemotongan' => $pemotongan,
                ]);
            }
            
            $akunPersediaan = Akun::where('nama_akun', 'Akun Persediaan')->first();
            $akunHutang = Akun::where('nama_akun', 'Akun Hutang')->first();
            $akunDiskon = Akun::where('nama_akun', 'Diskon Penjualan')->first();
            $akunPajak = Akun::where('nama_akun', 'PPN Keluaran')->first();

            if ($request->pemotongan && $request->akun_pemotongan) {
                $akunPemotongan = Akun::find($request->akun_pemotongan);

                // Debet ke Akun Pemotongan
                JurnalUmum::create([
                    'akun_id' => $akunPemotongan->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'pembelian_id' => $pembelian->id,
                    'debet' => $pemotongan,
                    'description' => 'Pemotongan Produk'
                ]);
            }

            if ($totalDiskon > 0) {
                // Debet ke Akun Diskon
                JurnalUmum::create([
                    'akun_id' => $akunDiskon->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'pembelian_id' => $pembelian->id,
                    'debet' => $totalDiskon,
                ]);
            }

            if ($totalPajak > 0) {
                // Kredit ke Akun PPN
                JurnalUmum::create([
                    'akun_id' => $akunPajak->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'pembelian_id' => $pembelian->id,
                    'kredit' => $totalPajak,
                ]);
            }

            // Debet ke Akun Persediaan
            JurnalUmum::create([
                'akun_id' => $akunPersediaan->id,
                'tanggal' => $request->tgl_transaksi,
                'supplier_id' => $request->supplier_id,
                'pembelian_id' => $pembelian->id,
                'debet' => $total - $pemotongan,
            ]);

            // Kredit ke Akun Hutang
            JurnalUmum::create([
                'akun_id' => $akunHutang->id,
                'tanggal' => $request->tgl_transaksi,
                'supplier_id' => $request->supplier_id,
                'pembelian_id' => $pembelian->id,
                'kredit' => $total - $pemotongan,
            ]);

            DB::commit();

            Helper::addUserLog('Menambah data ' .$this->aksesMenu . ' : ' . $pembelian->no_transaksi,$pembelian->toArray());
            
            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Menambah Data ' .$this->aksesMenu. ' : '.$pembelian->no_transaksi]);
        } catch (\Exception $e){
            DB::rollBack();
            dd($e->getMessage(), $request->all());
            return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : '.$e->getMessage()])->withInput($request->all());
        }
    }

    public function show(pembelian $pembelian)
    {
        if(!Helper::cek_akses($this->aksesMenu)){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'action'=>route($this->routeName.'.bayar',$pembelian->id),
        ];

        $produks = Produk::all();
        $suppliers = Supplier::all();

        $akuns = Akun::all();

        return view($this->viewName.'.show',compact('settings', 'produks', 'suppliers', 'akuns'),['data'=>$pembelian]);
    }
    
    public function edit(pembelian $pembelian)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Ubah')){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'action'=>route($this->routeName.'.update',$pembelian->id),
        ];

        $produks = Produk::all();
        $suppliers = Supplier::all();
        $akuns = Akun::all();
        

        return view($this->viewName.'.form',compact('settings', 'produks', 'suppliers', 'akuns'),['data'=>$pembelian]);
    }

    public function update(Request $request, pembelian $pembelian)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Ubah')){
            return abort(404);
        }
        
        $request->validate([
            'supplier_id'=>'required',
            'tgl_transaksi'=>'required',
            'tgl_jatuh_tempo'=>'required',
        ]);
        
        DB::beginTransaction();
        try {
            $old = $pembelian->toArray();
            
            $pembelian->update([
                'supplier_id'=>$request->supplier_id,
                'tgl_transaksi'=>$request->tgl_transaksi,
                'tgl_jatuh_tempo'=>$request->tgl_jatuh_tempo,
                'catatan'=>$request->catatan,
            ]);
            
            // Kembalikan Stok, Kemudian Hapus Detail
            for ($i=0; $i < count($pembelian->detail); $i++) { 
                $produk = Produk::find($request->produk_id[$i]);
                $produk->update([
                    'stok' => $produk->stok - $pembelian->detail[$i]->qty
                ]);
            }

            $total = 0;
            $totalDiskon = 0;
            $totalPajak = 0;

            $pembelian->detail()->delete();
             for ($i=0; $i < count($request->produk_id); $i++) { 
                $produk = Produk::find($request->produk_id[$i]);

                if(!$produk)
                    return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Produk Dengan Id ' . $request->produk_id[$i] . 'Tidak Ditemukan'])->withInput($request->all());

                    $diskon = 0;
                if ($request->tipe_diskon == 'rp') {
                    $diskon = $request->diskon[$i] ? Helper::onlyNumber($request->diskon[$i]) : 0;
                }elseif($request->tipe_diskon == 'persen'){
                    $harga = Helper::onlyNumber($request->harga_beli[$i]) * Helper::onlyFloat($request->qty[$i]);
                    $diskon = $request->diskon[$i] ? $harga * Helper::onlyNumber($request->diskon[$i]) / 100 : 0;
                }

                $subtotal = (Helper::onlyNumber($request->harga_beli[$i]) * Helper::onlyFloat($request->qty[$i])) - $diskon;

               $pajak = 0;
                if($request->pajak[$i]){
                    $pajak = ($subtotal * Helper::onlyNumber($request->pajak[$i]) / 100 );
                    $subtotal =  $subtotal + $pajak;
                }else if($request->pajak_lain[$i]) {
                    $pajak = ($subtotal * Helper::onlyNumber($request->pajak_lain[$i]) / 100 );
                    $subtotal=  $subtotal + $pajak;
                }

                $total += $subtotal;
                $totalDiskon += $diskon;
                $totalPajak += $pajak;
                
                $detail = DetailPembelian::create([
                    'pembelian_id' => $pembelian->id,
                    'produk_id' => $request->produk_id[$i],
                    'diskon' => $diskon,
                    'qty' => $request->qty[$i],
                    'harga' => Helper::onlyNumber($request->harga_beli[$i]),
                    'pajak' => ($request->pajak[$i]) ? Helper::onlyNumber($request->pajak[$i]) : null,
                    'pajak_lain' => ($request->pajak_lain[$i]) ? Helper::onlyNumber($request->pajak_lain[$i]) : null,
                ]);

                $produk->update([
                    'stok' => $produk->stok + $request->qty[$i]
                ]);

            }

            if(!$total)
                return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : Error, Total Gagal Dihitung '])->withInput($request->all());


            $pemotongan = 0;
            if ($request->pemotongan && $request->akun_pemotongan) {
                if ($request->tipe_pemotongan == 'rp') {
                    $pemotongan = Helper::onlyNumber($request->pemotongan);
                }else if($request->tipe_pemotongan == 'persen'){
                    $pemotongan = $total * Helper::onlyNumber($request->pemotongan) / 100;
                }
            }
            $pembelian->update([
                'total' => $total - $pemotongan,
                'pemotongan_akun_id' => $request->akun_pemotongan ?? NULL,
            ]);

            if ($request->pemotongan && $request->akun_pemotongan) {
                $pembelian->update([
                    'pemotongan' => $pemotongan,
                ]);
            }

            $akunPersediaan = Akun::where('nama_akun', 'Akun Persediaan')->first();
            $akunHutang = Akun::where('nama_akun', 'Akun Hutang')->first();
            $akunKas = Akun::where('nama_akun', 'Kas')->first();

            $pembelian->JurnalUmum()->delete();
            $akunDiskon = Akun::where('nama_akun', 'Diskon Penjualan')->first();
            $akunPajak = Akun::where('nama_akun', 'PPN Keluaran')->first();

            if ($request->pemotongan && $request->akun_pemotongan) {
                $akunPemotongan = Akun::find($request->akun_pemotongan);

                // Debet ke Akun Pemotongan
                JurnalUmum::create([
                    'akun_id' => $akunPemotongan->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'pembelian_id' => $pembelian->id,
                    'debet' => $pemotongan,
                    'description' => 'Pemotongan Produk'
                ]);
            }

            if ($totalDiskon > 0) {
                // Debet ke Akun Diskon
                JurnalUmum::create([
                    'akun_id' => $akunDiskon->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'pembelian_id' => $pembelian->id,
                    'debet' => $totalDiskon,
                ]);
            }

            if ($totalPajak > 0) {
                // Kredit ke Akun PPN
                JurnalUmum::create([
                    'akun_id' => $akunPajak->id,
                    'tanggal' => $request->tgl_transaksi,
                    'pelanggan_id' => $request->pelanggan_id,
                    'pembelian_id' => $pembelian->id,
                    'kredit' => $totalPajak,
                ]);
            }


            // Debet ke Akun Persediaan
            JurnalUmum::create([
                'akun_id' => $akunPersediaan->id,
                'tanggal' => $request->tgl_transaksi,
                'supplier_id' => $request->supplier_id,
                'pembelian_id' => $pembelian->id,
                'debet' => $total,
            ]);

            // Kredit ke Akun Hutang
            JurnalUmum::create([
                'akun_id' => $akunHutang->id,
                'tanggal' => $request->tgl_transaksi,
                'supplier_id' => $request->supplier_id,
                'pembelian_id' => $pembelian->id,
                'kredit' => $total,
            ]);

            DB::commit();

            Helper::addUserLog('Menambah data ' .$this->aksesMenu . ' : ' . $pembelian->no_transaksi,[
                'old' => $old,
                'update' => $pembelian->toArray()
            ]);

            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Mengubah Data ' .$this->aksesMenu. ' : '.$pembelian->no_transaksi]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Mengubah Data ' .$this->aksesMenu. ' : '.$e->getMessage()])->withInput($request->all());
        }
    }

    public function bayar(Request $request, $id)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Tambah')){
            return abort(404);
        }
        
        $request->validate([
            'akun_id'=>'required',
            'tgl_bayar'=>'required',
            'jumlah_bayar'=>'required',
        ]);
        
        DB::beginTransaction();
        try {
            $pembelian = Pembelian::find($id);
            
            $pembayaran_pembelian = PembayaranPembelian::create([
                'akun_id'=>$request->akun_id,
                'pembelian_id' => $pembelian->id,
                'tgl_bayar' => $request->tgl_bayar,
                'jumlah_bayar' => Helper::onlyNumber($request->jumlah_bayar),
            ]);

            if($pembelian->sisaTagihan > 0){
                $status = 'pending';
            }elseif($pembelian->sisaTagihan <= 0){
                $status = 'paid';
            }

            $pembelian->update([
                'status' => $status,
            ]);

            $akunKas = Akun::find($request->akun_id);
            $akunHutang = Akun::where('nama_akun', 'Akun Hutang')->first();

            // Debet ke Akun Hutang
            JurnalUmum::create([
                'akun_id' => $akunHutang->id,
                'tanggal' => $request->tgl_bayar,
                'supplier_id' => $pembelian->supplier_id,
                'pembayaran_pembelian_id' => $pembayaran_pembelian->id,
                'debet' => Helper::onlyNumber($request->jumlah_bayar),
            ]);

            // Kredit ke Akun Kas
            JurnalUmum::create([
                'akun_id' => $akunKas->id,
                'tanggal' => $request->tgl_bayar,
                'supplier_id' => $pembelian->supplier_id,
                'pembayaran_pembelian_id' => $pembayaran_pembelian->id,
                'kredit' => Helper::onlyNumber($request->jumlah_bayar),
            ]);

            DB::commit();

            Helper::addUserLog('Menambah data ' .$this->aksesMenu . ' Pembayaran : ' . $pembelian->no_transaksi,$pembelian->toArray());

            return redirect(route($this->routeName.'.show', $id))->with(['success'=>'Berhasil Menambah Data ' .$this->aksesMenu. ' Pembayaran: '.$pembelian->no_transaksi]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Mengubah Data ' .$this->aksesMenu. ' Pembayaran : '.$e->getMessage()])->withInput($request->all());
        }
    }

    public function update_detail(Request $request)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Ubah')){
            return abort(404);
        }

        DB::beginTransaction();
        try {
            
            $pembayaran_pembelian = PembayaranPembelian::find($request->id);

            $pembayaran_pembelian->update([
                'jumlah_bayar' => $request->harga
            ]);

            $akunKas = Akun::find($pembayaran_pembelian->akun_id);
            $akunHutang = Akun::where('nama_akun', 'Akun Hutang')->first();

            $jurnalUmumKas = JurnalUmum::where('pembayaran_pembelian_id', $pembayaran_pembelian->id)->where('akun_id', $akunKas->id)->first();

            $jurnalUmumHutang = JurnalUmum::where('pembayaran_pembelian_id', $pembayaran_pembelian->id)->where('akun_id', $akunHutang->id)->first();

            // Debet ke Akun Hutang
            $jurnalUmumHutang->update([
                'debet' => Helper::onlyNumber($request->harga),
            ]);

            // Kredit ke Akun Kas
            $jurnalUmumKas->update([
                'kredit' => Helper::onlyNumber($request->harga),
            ]);



            DB::commit();
            $data = [
                'status' => 200,
            ];
        } catch (\Exception $e) {
            DB::rollBack();
            $data = [
                'status' => 500,
                'message' => 'Tidak Dapat diubah',
                'data' => null,
            ];
        }
        return response()->json($data,$data['status']);
    }
    
    public function destroy(pembelian $pembelian)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Hapus')){
            return abort(404);
        }

        DB::beginTransaction();
        try {
            Helper::addUserLog('Menghapus data ' .$this->aksesMenu . ' : ' . $pembelian->no_transaksi,$pembelian->toArray());

            foreach ($pembelian->pembayaran as $pembayaran) {
                $pembayaran->delete();
            }
            $pembelian->delete();
        
            DB::commit();
            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Menghapus Data pembelian : '.$pembelian->no_transaksi]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Menghapus Data pembelian '.$pembelian->no_transaksi.' : '.$e->getMessage()]);
        }
    }
}
