<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\KategoriAkun;
use App\Models\Akun;
use Illuminate\Support\Facades\DB;
use App\Helpers\Helper;
use App\Models\JurnalUmum;

class AkunController extends Controller
{

    protected $routeName = 'akun';
    protected $viewName = 'akun';
    protected $menu = 'Master Data';
    protected $title = 'Chart Of Account';

    protected $aksesMenu = 'Chart Of Account';
    
    public function index()
    {
        if(!Helper::cek_akses($this->aksesMenu)){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'ubah'=>Helper::cek_akses($this->aksesMenu,'Ubah'),
            'hapus'=>Helper::cek_akses($this->aksesMenu,'Hapus')
        ];

        $getKategori = $_GET['kategori_id'] ?? null;
        if($getKategori){
            $datas = Akun::where('kategori_akun_id', $getKategori)->orderBy('kode_akun', 'ASC')->get();
        }else{
            $datas = Akun::orderBy('kode_akun', 'ASC')->get();
        }

        
        return view($this->viewName.'.index', compact('settings', 'datas'));
    }
    
    public function create()
    {
        if(!Helper::cek_akses($this->aksesMenu, 'Tambah')){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'action'=> route($this->routeName.'.store'),
        ];

        $kategoris = KategoriAkun::get();
        
        return view($this->viewName.'.form', compact('settings', 'kategoris'));
    }
    
    public function store(Request $request)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Tambah')){
            return abort(404);
        }

        $request->validate([
            'kode_akun'=>'required',
            'nama_akun'=>'required',
            'kategori_akun_id'=>'required',
        ]);

        DB::beginTransaction();

        try {
            $akun = Akun::create([
                'kode_akun'=>$request->kode_akun,
                'nama_akun'=>$request->nama_akun,
                'kategori_akun_id'=>$request->kategori_akun_id,
                'saldo'=>0,
            //    'no_telp'=>$request->no_telp,
            //    'alamat'=>$request->alamat,
            ]);

            DB::commit();

            Helper::addUserLog('Menambah data ' .$this->aksesMenu . ' : ' . $akun->nama_akun,$akun->toArray());
            
            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Menambah Data ' .$this->aksesMenu. ' : '.$akun->nama_akun]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : '.$e->getMessage()])->withInput($request->all());
        }
    }

    public function show(Akun $akun)
    {
        if(!Helper::cek_akses($this->aksesMenu)){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => 'Jurnal '. $akun->nama_akun .' - '. $this->title,
            'route' => $this->routeName,
        ];

        return view($this->viewName.'.show',compact('settings'),['data'=>$akun]);
    }
    
    public function edit(Request $request,$id)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Ubah')){
            return abort(404);
        }
        
        $settings = [
            'menu' => $this->menu,
            'title' => $this->title,
            'route' => $this->routeName,
            'action'=>route($this->routeName.'.update',$id),
        ];

        $kategoris = KategoriAkun::all();
        $akun = Akun::find($id);
        return view($this->viewName.'.form',compact('settings', 'kategoris'),['data'=>$akun]);
    }

    public function update(Request $request, Akun $akun)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Ubah')){
            return abort(404);
        }
        
        $request->validate([
            'kode_akun'=>'required',
            'nama_akun'=>'required',
            'kategori_akun_id'=>'required',
        ]);
        
        DB::beginTransaction();
        try {
            $old = $akun->toArray();
            $akun->update([
                'kode_akun'=>$request->kode_akun,
                'nama_akun'=>$request->nama_akun,
                'kategori_akun_id'=>$request->kategori_akun_id,
                'saldo'=>0,
            ]);
            DB::commit();

            Helper::addUserLog('Menambah data ' .$this->aksesMenu . ' : ' . $akun->nama_akun,[
                'old' => $old,
                'update' => $akun->toArray()
            ]);

            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Mengubah Data ' .$this->aksesMenu. ' : '.$akun->nama_akun]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Mengubah Data ' .$this->aksesMenu. ' : '.$e->getMessage()])->withInput($request->all());
        }
    }
    
    public function destroy(Request $request,Akun $akun)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Hapus')){
            return abort(404);
        }

        DB::beginTransaction();
        try {

            Helper::addUserLog('Menghapus data ' .$this->aksesMenu . ' : ' . $akun->nama_akun,$akun->toArray());

            $akun->delete();
        
            DB::commit();
            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Menghapus Data akun : '.$akun->nama_akun]);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Menghapus Data akun '.$akun->nama_akun.' : '.$e->getMessage()]);
        }
    }

    public function opening()
    {
        if(!Helper::cek_akses($this->aksesMenu, 'Tambah')){
            return abort(404);
        }

        $settings = [
            'menu' => $this->menu,
            'title' => 'Pembukaan Saldo ' . $this->title,
            'route' => $this->routeName,
            'action'=> route($this->routeName.'.openingStore'),
        ];

        $datas = Akun::all();
        
        return view($this->viewName.'.opening', compact('settings', 'datas'));
    }

    public function openingStore(Request $request)
    {
        if(!Helper::cek_akses($this->aksesMenu,'Tambah')){
            return abort(404);
        }

        $request->validate([
            'debet'=>'required',
            'kredit'=>'required',
        ]);

        DB::beginTransaction();

        try {

            $akuns = Akun::all();
            $ttl_debet = 0;
            $ttl_kredit = 0;
            for ($i=0; $i < count($request->debet); $i++) { 
                $ttl_debet += Helper::onlyNumber($request->debet[$i]);
                $ttl_kredit += Helper::onlyNumber($request->kredit[$i]);

                // Cari Opening Akun berdasarkan akun pada table jurnal umum
                $cari_jurnal = JurnalUmum::where('opening_akun', 'yes')->where('akun_id', $akuns[$i]->id)->first();

                if($cari_jurnal){
                    $cari_jurnal->update([
                        'debet'=> Helper::onlyNumber($request->debet[$i]),
                        'kredit'=> Helper::onlyNumber($request->kredit[$i]),
                    ]);
                }else{
                    JurnalUmum::create([
                        'tanggal'=>date('Y').'-01-01',
                        'akun_id'=>$akuns[$i]->id,
                        'opening_akun'=>'yes',
                        'debet'=> Helper::onlyNumber($request->debet[$i]),
                        'kredit'=> Helper::onlyNumber($request->kredit[$i]),
                    ]);
                }
            }

            // Cek, Jika ada selisih
            // Jika debet lebih besar dri kredit, maka tambahkan kredit di akun pembukaan saldo ekuitas
            $akunEkuitas = Akun::where('nama_akun', 'Pembukaan Saldo Ekuitas')->first();
            $cari_opening = JurnalUmum::where('opening_akun', 'yes')->where('akun_id', $akunEkuitas->id)->first();

            if($ttl_debet > $ttl_kredit){
                if($cari_opening){
                    $cari_opening->update([
                        'debet'=> 0,
                        'kredit'=> $ttl_debet,
                    ]);
                }else{
                    JurnalUmum::create([
                        'tanggal'=>date('Y').'-01-01',
                        'opening_akun' => 'yes',
                        'akun_id'=>$akunEkuitas->id,
                        'debet'=> 0,
                        'kredit'=> $ttl_debet,
                    ]);
                }

            // Jika kredit lebih besar dri debet, maka tambahkan debet di akun pembukaan saldo ekuitas
            }else if($ttl_kredit > $ttl_debet){
                if($cari_opening){
                    $cari_opening->update([
                        'debet'=> $ttl_kredit,
                        'kredit'=> 0,
                    ]);
                }else{
                    JurnalUmum::create([
                        'tanggal'=>date('Y').'-01-01',
                        'opening_akun' => 'yes',
                        'akun_id'=>$akunEkuitas->id,
                        'debet'=> $ttl_kredit,
                        'kredit'=> 0,
                    ]);
                }
            }else{
                // Jika tidak ada selisih, maka debet & kredit = 0
                if($cari_opening){
                    $cari_opening->update([
                        'debet'=> 0,
                        'kredit'=> 0,
                    ]);
                }else{
                    JurnalUmum::create([
                        'tanggal'=>date('Y').'-01-01',
                        'opening_akun' => 'yes',
                        'akun_id'=>$akunEkuitas->id,
                        'debet'=> 0,
                        'kredit'=> 0,
                    ]);
                }
            }

            DB::commit();

            Helper::addUserLog('Mengubah Pembukaan Saldo Ekuitas');
            
            return redirect(route($this->routeName.'.index'))->with(['success'=>'Berhasil Mengubah Saldo Awal ']);
        } catch (\Exception $e){
            DB::rollBack();
            return redirect()->back()->with(['error'=>'Gagal Menambah Data ' .$this->aksesMenu. ' : '.$e->getMessage()])->withInput($request->all());
        }
    }
}
